# Plugin for TWiki Enterprise Collaboration Platform, http://TWiki.org/
#
# Copyright (C) 2014-2015 Wave Systems Corp.
# Copyright (C) 2014-2015 Peter Thoeny, peter[at]thoeny.org 
# Copyright (C) 2014-2015 TWiki Contributors. All Rights Reserved.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version. For
# more details read LICENSE in the root of this distribution.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details, published at
# http://www.gnu.org/copyleft/gpl.html
#
# As per the GPL, removal of this notice is prohibited.

package TWiki::Plugins::WhereIsPlugin::Core;

# =========================
sub new {
    my ( $class, $debug ) = @_;

    my $this = {
          Debug          => $debug,
          LastSeenMsg    => $TWiki::cfg{Plugins}{WhereIsPlugin}{LastSeenMsg}
                         || '$spacedname last seen at $location $ago',
          NotSeenMsg     => $TWiki::cfg{Plugins}{WhereIsPlugin}{NotSeenMsg}
                         || '$spacedname is unknown or has not been seen',
          WorkAreaDir    => TWiki::Func::getWorkArea( 'WhereIsPlugin' ),
        };
    bless( $this, $class );
    $this->_writeDebug( "new: constructor" );

    return $this;
}

# =========================
sub VarWHEREIS
{
    my ( $this, $params ) = @_;
    my $action  = $params->{_DEFAULT} || $params->{action} || 'showlocation';
    $this->_writeDebug( "VarWHEREIS($action)" );
    my $text = '';
    if( $action eq 'recordlocation' ) {
        $text = $this->_recordLocation();
    } elsif( $action eq 'setlocation' ) {
        $text = $this->_setLocation( $params->{address}, $params->{location} );
    } elsif( $action eq 'showlocation' ) {
        $text = $this->_showLocation( $params->{user}, $params->{lastseenmsg}, $params->{notseenmsg} );
    } elsif( $action eq 'editlocations' ) {
        $text = $this->_editLocations( $params->{limit} );
    } elsif( $action eq 'showrecent' ) {
        $text = $this->_showRecent( $params->{limit}, $params->{header}, $params->{format} );
    } elsif( $action ) {
        $text = "WHEREIS ERROR: Unrecogized action parameter $action";
    } else {
        $text = '';
    }
    if( $this->{Debug} ) {
        $text =~ /^(.{0,80})/s;
        $this->_writeDebug( "VarWHEREIS($action) return:\n$1" );
    }
    return $text;
}

# =========================
sub _recordLocation
{
    my ( $this ) = @_;

    $this->_writeDebug( "_recordLocation()" );
    my $text = '';
    if( TWiki::Func::getContext()->{authenticated} ) {
        my $addr = $ENV{REMOTE_ADDR} || '127.0.0.1';
        my $userData = {
            addr     => $addr,
            location => $this->_addrToLocation( $addr, 0 ),
            atime    => time(),
        };
        unless( $userData->{location} ) {
            $text = TWiki::Func::readTemplate( 'whereisdialog' );
            $text =~ s/\%WHEREISLOCATION\%/$this->_addrToLocation( $addr, 1 )/geo;
        }
        $this->_saveData( 'user', TWiki::Func::getWikiName(), $userData );
    }
    return $text;
}

# =========================
sub _setLocation
{
    my ( $this, $addr, $location ) = @_;

    $addr ||= '';
    $addr =~ s/[^0-9\.]//go;
    $addr ||= $ENV{REMOTE_ADDR} || '127.0.0.1';
    return 'ERROR: Invalid IP address' unless( $addr =~ /^[0-9]{1,3}(\.[0-9]{1,3}){1,3}$/ );

    $location ||= '';
    $location =~ s/[\n\r]//go;
    $location =~ m/^(.{0,100}).*$/;
    $this->_writeDebug( "_setLocation('$location') for $addr" );

    if( $location eq 'DELETE' ) {
        $this->_deleteDataRecord( 'addr', $addr );

    } else {
        my $addrData = {
            name     => $location,
        };
        $this->_saveData( 'addr', $addr, $addrData );
    }

    return 'OK';
}

# =========================
sub _showLocation
{
    my ( $this, $user, $lastSeenMsg, $notSeenMsg ) = @_;

    $user ||= TWiki::Func::getWikiName(); 
    $this->_writeDebug( "_showLocation()" );
    my $text = '';
    my $location = '';
    my $ago = '';
    my $userData = $this->_readData( 'user', $user );
    if( $userData && $userData->{atime} ) {
        $location = $this->_addrToLocation( $userData->{addr}, 1 );
        $ago = _timeDiff( $userData->{atime}, time() );
        $text = $lastSeenMsg || $this->{LastSeenMsg};
    } else {
        $text = $notSeenMsg || $this->{NotSeenMsg};
    }
    $text =~ s/\$wikiname/$user/go;
    $text =~ s/\$wikiusername/'%USERSWEB%.' . $user/geo;
    $text =~ s/\$spacedname/\%CALCULATE{\$PROPERSPACE($user)}\%/go;
    $text =~ s/\$location/$location/go;
    $text =~ s/\$ago/$ago/go;
    return $text;
}

# =========================
sub _editLocations
{
    my ( $this, $limit ) = @_;
    $limit ||= 100;
    my $authenticated = TWiki::Func::getContext()->{authenticated};
    my $isAdmin = TWiki::Func::isAnAdmin();
    my $wikiName = TWiki::Func::getWikiName();
    my ( $start, $header, $format, $addlocation, $end ) =
      split( '%SPLIT%', TWiki::Func::readTemplate( 'whereiseditlocations' ) );
    $this->_writeDebug( "_editLocations($limit)" );
    my $text = "$start$header";
    my @userData = sort{ $a->{id} cmp $b->{id} } $this->_searchData( 'user', 'name' );
    my @addrData = sort{ $a->{name} cmp $b->{name} } $this->_searchData( 'addr', 'name' );
    foreach my $data ( @addrData ) {
        last unless( $limit-- );
        my $id = $data->{id};
        my $addr = $id;
        $id =~ s/[^\w]/_/go;
        my $location = $data->{name};
        my @users = map{ $_->{id} } grep{ $_->{addr} eq $addr } @userData;
        my $line = $format;
        my $saveOK = ( $authenticated && ( $isAdmin || grep{ $_ eq $wikiName } @users ) ) ? 1 : 0;
        $line =~ s/\%WHEREISSAVEOK\%/$saveOK/go;
        $line =~ s/\%WHEREISADDRESS\%/$addr/go;
        $line =~ s/\%WHEREISLOCATION\%/$location/go;
        $line =~ s/\%WHEREISENCODEDADDRESS\%/_entityEncode( $addr, " \n\r" )/geo;
        $line =~ s/\%WHEREISENCODEDLOCATION\%/_entityEncode( $location, " \n\r" )/geo;
        $line =~ s/\%WHEREISUSERS\%/join( ', ', map{ '%USERSWEB%.' . $_ } @users )/geo;
        $text .= $line;
    }
    if( $authenticated && $isAdmin ) {
        $addlocation =~ s/\%WHEREISSAVEOK\%/1/go;
        $text .= $addlocation;
    }
    $text .= $end;
    return $text;
}

# =========================
sub _showRecent
{
    my ( $this, $limit, $header, $format ) = @_;
    $limit ||= 20;
    $header ||= '| *User* | *At location* | *Last seen* |';
    $format ||= '| $wikiusername | $location |  $ago |';
    $this->_writeDebug( "_showRecent($limit)" );
    my $text = "$header\n";
    my @usersData = sort{ $b->{atime} <=> $a->{atime} } $this->_searchData( 'user', 'atime' );
    foreach my $data ( @usersData ) {
        last unless( $limit-- );
        my $user = $data->{id};
        my $location = $this->_addrToLocation( $data->{addr}, 1 );
        my $ago = _timeDiff( $data->{atime}, time() );
        my $line = $format;
        $line =~ s/\$wikiname/$user/go;
        $line =~ s/\$wikiusername/'%USERSWEB%.' . $user/geo;
        $line =~ s/\$spacedname/\%CALCULATE{\$PROPERSPACE($user)}\%/go;
        $line =~ s/\$location/$location/go;
        $line =~ s/\$ago/$ago/go;
        $text .= "$line\n";
    }
    return $text;
}

# =========================
sub _addrToLocation {
    my ( $this, $addr, $handleUnknownLocation ) = @_;
    $this->_writeDebug( "_addrToLocation($addr)" );

    my $data = $this->_readData( 'addr', $addr );
    unless( $data ) {
        my $sub = $addr;
        $sub =~ s/^(.*)\..*/$1/; # Try sub-net by cutting 'x' from 'a.b.c.x'
        $data = $this->_readData( 'addr', $sub );
    }
    unless( $data ) {
        my $sub = $addr;
        $sub =~ s/^(.*)(\.[^\.]+){2}/$1/; # Try sub-net by cutting 'x.x' from 'a.b.x.x'
        $data = $this->_readData( 'addr', $sub );
    }
    return $data->{name} if( $data && $data->{name} );
    return '' unless( $handleUnknownLocation );

    my $location = '';
    if( TWiki::Func::getContext()->{'GeoLookupPluginEnabled'} ) {
        # GeoLookupPlugin is enabled, figure out location by geolocation lookup
        my $geo = TWiki::Func::expandCommonVariables( '%GEOLOOKUP{' . $addr . '}%' );
        $geo =~ s/USA$//; # do not show USA, just state in USA
        $geo =~ s/,//g;
        $geo =~ s/^ *//;
        $geo =~ s/ *$//;
        $location = "unspecified location in $geo" if( $geo && $geo !~ /GEOLOOKUP/ );
    }
    $location ||= 'unknow location';
    return $location;
}

# =========================
sub _searchData {
    my ( $this, $type, $key, $value ) = @_;

    my @dataArray = ();
    return $dataArray unless( $type && $key );

    if( opendir( DIR, $this->{WorkAreaDir} ) ) {
        my @files = sort
                    grep { /^$type-.*\.txt$/ }
                    readdir(DIR);
        closedir DIR;
        for my $id ( @files ) {
            $id =~ s/^$type-(.*)\.txt$/$1/;
            my $data = $this->_readData( $type, $id );
            my $v = $data->{$key};
            if( !$value || ( $v && $v eq $value ) ) {
                push( @dataArray, $data );
            }
        }
    }
    return @dataArray;
}

# =========================
sub _readData
{
    my ( $this, $type, $id ) = @_;
    my $data;
    return $data unless( $type && $id );
    $this->_writeDebug( "_readData($type, $id)" );
    my $fileName = $this->{WorkAreaDir} . "/$type-$id.txt";
    my $found = 0;
    foreach my $line ( split( /[\n\r]/, TWiki::Func::readFile( $fileName ) ) ) {
        if( $line =~ /^([\w]+): ([^\n\r]*)/ ) {
            $data->{$1} = $2;
            $found = +1;
        }
    }
    $data->{id} = $id if( $found );
    # use Data::Dumper;
    # $this->_writeDebug( Dumper($data) );
    return $data;
}

# =========================
sub _saveData
{
    my ( $this, $type, $id, $data ) = @_;
    return unless( $type && $id );
    $this->_writeDebug( "_saveData($type, $id)" );
    my $fileName = $this->{WorkAreaDir} . "/$type-$id.txt";
    $fileName =~ /^(.*)$/;
    $fileName = $1; # untaint
    my $text = "# This file is generated, do not modify\n";
    foreach my $key ( sort grep { !/^id$/ } keys %{$data} ) {
        $text .= "$key: " . $data->{$key} . "\n";
    }
    #$this->_writeDebug( " \n===== Save $type data of $id =====:\n$text=====end=====" );
    TWiki::Func::saveFile( $fileName, $text );
}

# =========================
sub _deleteDataRecord
{
    my ( $this, $type, $id ) = @_;
    return unless( $type && $id );
    $this->_writeDebug( "_deleteDataRecord($type, $id)" );
    my $fileName = $this->{WorkAreaDir} . "/$type-$id.txt";
    $fileName =~ /^(.*)$/;
    $fileName = $1; # untaint
    unlink( $fileName );
}

# =========================
sub _writeDebug
{
    my ( $this, $msg ) = @_;
    return unless( $this->{Debug} );
    TWiki::Func::writeDebug( "- WhereIsPlugin::Core::$msg" );
}

# =========================
sub _entityEncode
{
    my ( $text ) = @_;
    if( $TWiki::Plugins::VERSION >= 6.0 ) {
        TWiki::Func::entityEncode( @_ );
    } else {
        TWiki::entityEncode( @_ );
    }
}

# =========================
sub _timeDiff
{
    my ( $old, $new ) = @_;
    my $timeDiff = $new - $old;

    if( $timeDiff < 60 ) {
        # if less than one minute
        return 'moments ago';
    }

    # code borrowed from FORMATTIMEDIFF of SpreadSheetPlugin:
    my $prec = 0;
    my @unit  = ( 0, 0, 0, 0, 0, 0 ); # sec, min, hours, days, month, years
    my @factor = ( 1, 60, 60, 24, 30.4166, 12 ); # sec, min, hours, days, month, years
    my @singular = ( 'second',  'minute',  'hour',  'day',  'month',  'year' );
    my @plural =   ( 'seconds', 'minutes', 'hours', 'days', 'months', 'years' );
    $unit[0] = $timeDiff;
    my @arr = ();
    my $i = 0;
    my $val1 = 0;
    my $val2 = 0;
    for( $i = 0; $i < 5; $i++ ) {
        $val1 = int($unit[$i]);
        $val2 = $unit[$i+1] = int($val1 / $factor[$i+1]);
        $val1 = $unit[$i] = $val1 - int($val2 * $factor[$i+1]);

        push( @arr, "$val1 $singular[$i]" ) if( $val1 == 1 );
        push( @arr, "$val1 $plural[$i]" )   if( $val1 > 1 );
    }
    push( @arr, "$val2 $singular[$i]" ) if( $val2 == 1 );
    push( @arr, "$val2 $plural[$i]" )   if( $val2 > 1 );
    push( @arr, "0 $plural[0]" )    unless( @arr );
    my @reverse = reverse( @arr );
    $#reverse = $prec if( @reverse > $prec );
    my $result = join( ', ', @reverse );
    $result =~ s/(.+)\, /$1 and /;
    $result .= ' ago';
    return $result;
}

# =========================
1;
