# Module of TWiki Enterprise Collaboration Platform, http://TWiki.org/
#
# Copyright (C) 2014 Wave Systems Corp.
# Copyright (C) 2014-2021 Peter Thoeny, peter[at]thoeny.org 
# Copyright (C) 2014-2021 TWiki Contributors. All Rights Reserved.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 3
# of the License, or (at your option) any later version. For
# more details read LICENSE in the root of this distribution.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
# As per the GPL, removal of this notice is prohibited.

=pod

---+ package TWiki::LoginManager::SmsTwoStepAuth

=cut

package TWiki::LoginManager::SmsTwoStepAuth;

use strict;
use Assert;
use Error qw( :try );

our $sessionVarName = '_SmsTwoStepAuthAccessCode';

=pod

---++ ClassMethod new ($session, $impl)

Construct the two-step auth management object

=cut

sub new {
    my ( $class, $twiki ) = @_;
    my $this = bless( { twiki => $twiki }, $class );

    return $this;
}

=begin twiki

---++ ObjectMethod finish()

Break circular references.

=cut

# Note to developers; please undef *all* fields in the object explicitly,
# whether they are references or not. That way this method is "golden
# documentation" of the live fields in the object.
sub finish {
    my $this = shift;
    undef $this->{twiki};
}

=pod

---++ ObjectMethod secondStepAuth()

Do second step authentication:

   * Send user an SMS with one time access code
   * Show dialog to enter access code

=cut

sub secondStepAuth {
    my( $this, $loginName, $origUrl ) = @_;
    my $session = $this->{twiki};
    my $wikiName = $session->{users}->getWikiName( $loginName );
    my $debug = $TWiki::cfg{SmsTwoStepAuthContrib}{Debug};
    my $debugID = "TWiki::LoginManager::SmsTwoStepAuth::secondStepAuth";

    # skip second auth if in command line context
    return '' if $session->inContext( 'command_line' );

    # check two-step auth mode
    my( $meta, $text ) = TWiki::Func::readTopic( $TWiki::cfg{UsersWebName}, $wikiName );
    my $twoStepAuth = $TWiki::cfg{SmsTwoStepAuthContrib}{TwoStepAuth} || 'required';
    if( $twoStepAuth eq 'disabled' ) {
        TWiki::Func::writeDebug( "$debugID: Two-step auth disabled for all users" ) if( $debug );
        return ''; # use single setp auth

    } elsif( $twoStepAuth eq 'optional' ) {
        my $field = $meta->get( 'FIELD', 'TwoStepAuth' );
        unless( $field && TWiki::Func::isTrue( $field->{value} ) ) {
            TWiki::Func::writeDebug( "$debugID: $wikiName opted out of two-step auth" ) if( $debug );
            return ''; # user did not opt in for two-step auth
        }

    } else {
        # else continue, two-step auth is required
    }

    # check whitelist
    my $addr = $session->{request}->remoteAddress() || '';
    my $whitelist = $TWiki::cfg{SmsTwoStepAuthContrib}{WhitelistAddresses} || '';
    $whitelist =~ s/[^0-9\.\,]//g;
    $whitelist =~ s/,/\|/g;
    if( $addr && $whitelist && $addr =~ /^($whitelist)/ ) {
        # trusted environment, skip second step authentication
        TWiki::Func::writeDebug( "$debugID: Whitelisted IP for $wikiName, single step auth" )
          if( $debug );
        return '';
    }

    # two-step authentication is required - generate one-time-use access code
    my $accessCode = 'ac' . sprintf( '%02d', rand(100) ) . '-' . sprintf( '%04d', rand(10000) );

    # initialize variables
    my $allowEmail      = $TWiki::cfg{SmsTwoStepAuthContrib}{AllowEmail} || '';
    my $messageTemplate = $TWiki::cfg{SmsTwoStepAuthContrib}{SmsMessageTmpl} || 'smstwostepmessage';
    my $dialogTemplate  = $TWiki::cfg{SmsTwoStepAuthContrib}{SmsLoginTmpl}   || 'smstwosteplogin';
    my $mobile  = '';
    my $carrier = '';
    my $email   = '';
    my $filter  = '';
    my $authPossible = 1;

    # get mobile number and carrier
    my $field = $meta->get( 'FIELD', 'Email' );
    $email = $field->{value} if( $field );
    $field = $meta->get( 'FIELD', 'Mobile' );
    $mobile = $field->{value} if( $field );
    $field = $meta->get( 'FIELD', 'MobileCarrier' );
    $carrier = $field->{value} if( $field );

    # get gateway e-mail from mobile carrier table row based on user's Mobile Carrier field
    ( $meta, $text ) = TWiki::Func::readTopic( $TWiki::cfg{SystemWebName}, 'SmsTwoStepAuthContrib' );
    # Example mobile carrier table row:
    #   | *Type* | *Carrier* | *E-mail* | *Filter* | *Activation* |
    #   | E2SMS | USA: AT&T | $phone@txt.att.net | ^\+?1? | |
    my $gatewayEmail = '';
    if( $carrier && $text =~ /.*E2SMS *\| *$carrier *\| *(.*?) *\| *(.*?) *\|/ ) {
        $gatewayEmail = $1;
        $filter = $2;
    }

    # compose SMS e-mail address from Mobile field and from gateway e-mail
    if( $mobile && $carrier && $gatewayEmail ) {
        $mobile =~ s/$filter//;
        $mobile =~ s/[^0-9]//g;
        $gatewayEmail =~ s/\$phone/$mobile/g;
        $email = $gatewayEmail;

    } elsif( $allowEmail eq '1'
            || ( $allowEmail && grep { /^$wikiName$/ } split( /, */, $allowEmail ) ) ) {
        # send to user's registered e-mail address
        my @emails = map{ my $a = "$wikiName <$_>"; $a; }
                     $session->{users}->getEmails( $loginName );
        if( scalar @emails ) {
            # use system e-mail, overriding Email form field on user profile topic
            $email = join( ', ', @emails );
        } else {
            # system e-mail not available, use e-mail on user profile topic
            $email = "$wikiName <$email>";
        }
        $messageTemplate = $TWiki::cfg{SmsTwoStepAuthContrib}{EmailMessageTmpl}
                        || 'smstwostepemailmessage';
        $dialogTemplate  = $TWiki::cfg{SmsTwoStepAuthContrib}{EmailLoginTmpl}
                        || 'smstwostepemaillogin';

    } else {
        # insufficient credentials, user can't log in
        $authPossible = 0;
        $dialogTemplate  = $TWiki::cfg{SmsTwoStepAuthContrib}{ErrorLoginTmpl}
                        || 'smstwosteperrorlogin';
    }
    if( $debug ) {
        TWiki::Func::writeDebug( "$debugID debug for $wikiName:\n"
          . "allowEmail: $allowEmail\n"
          . "messageTemplate: $messageTemplate\n"
          . "dialogTemplate: $dialogTemplate\n"
          . "mobile: $mobile\n"
          . "carrier: $carrier\n"
          . "email: $email\n"
          . "filter: $filter\n"
          . "authPossible: $authPossible" );
    }

    # save access code for later verification
    my $now = time();
    TWiki::Func::setSessionValue( $sessionVarName, "$accessCode:$now:$loginName" );

    # send e-mail to log-in user with access code
    if( $authPossible ) {
        my $tmpl = $session->templates->readTemplate( $messageTemplate, $session->getSkin() );
        return "Two-step authentication installation error: $messageTemplate template not found"
          unless( $tmpl );
        $tmpl =~ s/%EMAILADDRESS%/$email/geo;
        $tmpl =~ s/%ACCESSCODE%/$accessCode/go;
        $tmpl = $session->handleCommonTags( $tmpl, $session->{webName}, $session->{topicName} );
        $tmpl =~ s/<nop>//g;
        if( $debug ) {
            TWiki::Func::writeDebug( "$debugID e-mail:" );
            TWiki::Func::writeDebug( "===( START )=============" );
            TWiki::Func::writeDebug( "$tmpl" );
            TWiki::Func::writeDebug( "===(  END  )=============" );
        }
        my $warnings = $session->net->sendEmail( $tmpl );
        return "$warnings <hr /><pre>$tmpl</pre>" if( $warnings );
    }

    # load and return "enter access code" template
    my $tmpl = $session->templates->readTemplate( $dialogTemplate, $session->getSkin() )
            || "Two-step authentication installation error: $dialogTemplate template not found";
    $tmpl =~ s/%LOGINNAME%/$loginName/go;
    $tmpl =~ s/%ORIGURL%/$origUrl/go;
    return $tmpl;
}

=pod

---++ ObjectMethod verifyAuth()

Verify access code on second step authentication. Return empty string if OK, else return error string.

=cut

sub verifyAuth {
    my( $this, $loginName, $accessCode ) = @_;
    my $session = $this->{twiki};
    my $debug = $TWiki::cfg{SmsTwoStepAuthContrib}{Debug};
    my $debugID = "TWiki::LoginManager::SmsTwoStepAuth::verifyAuth";

    return '' if $session->inContext( 'command_line' );

    # compare to saved access code
    my ( $expectedAC, $timestamp, $expectedLN )
      = split( /:/, TWiki::Func::getSessionValue( $sessionVarName ), 3 );
    TWiki::Func::clearSessionValue( $sessionVarName ); # clear session variable (one time use only)
    my $maxAge = $TWiki::cfg{SmsTwoStepAuthContrib}{MaxAge} || 600;
    my $error = '';
    unless( $accessCode
            && ( $accessCode eq $expectedAC )
            && ( $loginName eq $expectedLN ) 
            && $timestamp + $maxAge > time()
    ) {
        $error = $TWiki::cfg{SmsTwoStepAuthContrib}{AcessCodeError} ||
                 'Invalid or outdated access code, please try again';
    }
    if( $TWiki::cfg{SmsTwoStepAuthContrib}{Debug} ) {
        TWiki::Func::writeDebug( "TWiki::LoginManager::SmsTwoStepAuth::verifyAuth return: '$error'" );
    }
    return $error;
}

1;
