#!/usr/bin/perl -w 

# --- Testing the SapConnectPlugin

use TWiki::Plugins::SapConnectPlugin;

use Test::More tests => 26;

init_tests();

# Test API function 
is(    get_data_from_sap( type => 'transports', dest => 'D12' ),
       'http://d12.migros.ch:1091/sap/bc/twiki/transports',
       'get_data_from_sap() URL with no id is built properly' );
is(    get_data_from_sap( type => 'transport', id => 'D12K123456' ),
       'http://d11.migros.ch:1090/sap/bc/twiki/transport?id=D12K123456',
       'get_data_from_sap() URL with id is built properly' );
is(    get_data_from_sap( type => 'transports', args => { list_scope => 'light' } ),
       'http://d11.migros.ch:1090/sap/bc/twiki/transports?list_scope=light',
       'get_data_from_sap() URL with extra parameters is built properly' );

# Test request URL properly produced
is(    incl_from_sap( 'docu', {_DEFAULT => 'FU.Z_TEST', dest => 'D12'}),
       'http://d12.migros.ch:1091/sap/bc/twiki/docu?id=FU.Z_TEST',
       'incl_from_sap([docu]) generating the correct URL' );

is(    (incl_from_sap( 'code', {_DEFAULT => 'FU.Z_TEST'})=~ /<pre>\s*(.*?)\s*<\/pre>/ )[0],
       'http://d11.migros.ch:1090/sap/bc/twiki/code?id=FU.Z_TEST',
       'incl_from_sap([code]) generating the correct URL' );
is(    incl_from_sap( 'addi', {_DEFAULT => 'TRANSPORT.D12K165202'}),
       'http://d11.migros.ch:1090/sap/bc/twiki/transport?id=D12K165202',
       'incl_from_sap([addi]) generating the correct URL' );

is(    url_for_sap( 'docu', { _DEFAULT => 'CL.ZCL_TEST',
                              expand   => 'true' } ),
       'http://sapdev.mits.ch/twiki/bin/view/TWiki/SapDocu?id=CL.ZCL_TEST&expand=X',
       'url_for_sap() generates URL to special TWiki topic' );
       

# enrich URL from TWiki tag args
is(    test_enrich_url({ expand     => 'true',
                         _DEFAULT => 'RE.ZZ_TEST' })->query_param('expand'),
       'X', 
       'enrich_url_from_tag_args() maps true values to X' );
is(    test_enrich_url({ expand   => 'false',
                         _DEFAULT => 'RE.ZZ_TEST' })->query_param('expand'),
       ' ', 
       'enrich_url_from_tag_args() maps false values to space' );
is(    test_enrich_url({ level     => 3,
                         _DEFAULT  => 'RE.ZZ_TEST' })->query_param('level'),
       '3', 
       'enrich_url_from_tag_args() maps level value' );


is(    test_enrich_url({ _DEFAULT => 'RE.ZZ_TEST' })->path,
       '/docu', 
       'enrich_url_from_tag_args([docu]) sets path extension "docu"' );
is(    test_enrich_url({ _DEFAULT => 'RE.ZZ_TEST' })->query_param('id'),
       'RE.ZZ_TEST', 
       'enrich_url_from_tag_args([docu]) sets ID parameter from default argument' );

is(    test_enrich_url({ _DEFAULT => 'TRANSPORT.D12K104256'},'addi')->path,
       '/transport', 
       'enrich_url_from_tag_args([addi]) fills path extension from first part of default argument' );
is(    test_enrich_url({ _DEFAULT => 'TRANSPORT.D12K104256' },'addi')->query_param('id'),
       'D12K104256', 
       'enrich_url_from_tag_args([addi]) sets ID parameter from 2nd part of default argument' );

sub test_enrich_url {
  my $url = URI->new( 'http://test.com' );
  my $args = shift;
  my $type = shift || 'docu';
  enrich_url_from_tag_args( $url, $type, $args);
  return $url;
  }

# get URL function
isa_ok( get_url_from_dest(),      'URI',     
        'get_url_from_dest() passes URI object');
like(   get_url_from_dest(),      qr/d11/,   
        'get_url_from_dest() uses default dest if none passed');
is(     get_url_from_dest('D12'), 'http://d12.migros.ch:1091/sap/bc/twiki/', 
        'get_url_from_dest() passes complete URL');
like(   get_url_from_dest('d11'), qr/d11/,'get_url_from_dest() accepts lowercase destinations' );
fails(  get_url_from_dest,      'D99', 
        'get_url_from_dest() raises exception for unknown destinations');
  
for (qw/
  RE.ZZ_TEST
  CL.ZCL_XML_HELPER
  TX.ZZ_HELP
  fu.z_convert_itf
  /) { 
  passes( \&check_docu_key, $_, "is_docu_key() accepts syntactically correct arguments");
  } 
  
for (qw/
  REP.ZZ_TEST
  RE
  Something
  /) {
  fails( \&check_docu_key, $_, "is_docu_key() refuses syntactically incorrect arguments");
  }
  
sub fails {
  my $testname = pop @_;
  ok( died( @_ ), $testname );
  }  

sub passes {
  my $testname = pop @_;
  ok( !died( @_ ), $testname );
  }  

sub died {
  my ($f,@args) = @_;
  eval {
    &$f(@args);
    };
  return !!$@;
  }

sub init_tests {
  no warnings 'once';
  # initialize some variables
  %TWiki::Plugins::SapConnectPlugin::sapDest = ( D11 => "d11.migros.ch:1090",
                                                 D12 => "d12.migros.ch:1091" );
  $TWiki::Plugins::SapConnectPlugin::defaultDest = "D11";
  $TWiki::Plugins::SapConnectPlugin::sapPath = "/sap/bc/twiki/";
  $TWiki::Plugins::SapConnectPlugin::scheme = "http";
  no warnings;
  *TWiki::Plugins::SapConnectPlugin::get_request = sub { return shift };             
  *TWiki::Plugins::SapConnectPlugin::get_topic_url_for_link = sub { return URI->new( 'http://sapdev.mits.ch/twiki/bin/view/TWiki/Sap' . ucfirst lc shift ); };

# For better readability of the tests: Import these methods into own namespace  
  import_methods_under_test(qw|
    incl_from_sap
    url_for_sap
    check_docu_key
    enrich_url_from_tag_args
    get_url_from_dest
    get_data_from_sap
    |);
  }

sub import_methods_under_test {
  for (@_) {
    *{$_} = \*{"TWiki::Plugins::SapConnectPlugin::$_"};
    }
  }