# SapConnectPlugin 
# Get documentation or code from an SAP system
# The information can be made available as a link, or directly embedded into the topic
# See Plugin topic for history and plugin information

package TWiki::Plugins::SapConnectPlugin;

use strict;
use warnings;

# Can be activated for development if running Twiki on Apache w/ mod_perl
# use Apache2::Reload;  

# Exports an API function for access in scripts or other modules
use base qw(Exporter);
our @ISA = qw(Exporter);
our @EXPORT_OK = qw(get_data_from_sap);

# For building URLs
use URI;
use URI::QueryParam;

# For performing the HTTP request
use LWP::UserAgent;

use TWiki::Func;

our ($VERSION,$pluginName);

# Data for the SAP connections
our ($scheme,%sapDest,$sapPath,$defaultDest);

# Parameters which will be routed to SAP
my @ROUTED_PARAMETERS;

BEGIN {

# Version number, to avoid conflicts in initPlugin()  
  $VERSION = '1.000';
  
# Plugin name should always be defined
  $pluginName = 'SapConnectPlugin';

# Parameters which will be routed to SAP
  @ROUTED_PARAMETERS = qw/level noheader expand/;

  }

# API function for usage in other modules or scripts
sub get_data_from_sap {
  my %params = @_;

# External calls have to load the system settings first
  readSapDestinations() unless $defaultDest;

  my $dest   = $params{dest} || $defaultDest;
  my $id     = $params{id}   || "";
  my $args   = $params{args} || {};
  my $type   = $params{type} or die "Please specify request type\n";  

  $args->{_DEFAULT} = $id ? "$type.$id" : $type; 
    
  my $url = get_url_from_dest($params{dest});
  enrich_url_from_tag_args( $url, 'addi', $args );    
  return get_request( $url );  
  
  }

# TWiki hook "initPlugin"
sub initPlugin {

# Read the SAP destinations from Main::TWikiPreferences
  readSapDestinations( );

# No version conflicts: Everything OK
  return 1;

  }

# TWiki hook "commonTagsHandler"
sub commonTagsHandler {

# Common Syntax of all tags handled by this plugin

# INCLSAPDOCU - Include SAP documentation into TWiki topic  
# INCLSAPCODE - Include SAP code into TWiki topic
# URLSAPDOCU  - Only provide a hyperlink to SAP docu topic
# URLSAPCODE  - Only provide a hyperlink to SAP code topic
# INCLSAPADDI - Additional function 
  $_[0] =~ s/%(INCL|URL)SAP(DOCU|CODE|ADDI)\{(.*?)\}%/handle_single_tag($1,$2,$3)/ge ;
    
  }


sub handle_single_tag {

  my $handler  = ( $_[0] eq 'INCL' ) ? \&incl_from_sap : \&url_for_sap;
  my $type     = lc $_[1];
  my %tag_args = TWiki::Func::extractParameters( $_[2] );
  
  my $result = eval {
    local $SIG{__DIE__};   # avoiding interference with CGI::Carp
    &$handler( $type, \%tag_args );
    };
  return errorMsg($@) if $@;
  return $result;
  
  }


# Include TWiki content from SAP 
sub incl_from_sap {
  
  my ($type,$tag_args) = @_;
    
  my $url = get_url_from_dest($tag_args->{dest});

  enrich_url_from_tag_args( $url, $type, $tag_args );  
  
  my $result = get_request( $url );  
  preformat( \$result ) if $type eq 'code';
  
  return $result;

  }

# Generates an URL for docu or code display  
sub url_for_sap {  

  my ($type,$tag_args) = @_;

  my $url = get_topic_url_for_link( $type );
  enrich_url_from_tag_args( $url, '', $tag_args);
  
  return $url->as_string;

  }  


# Add parameters from TWiki tag args to URL
sub enrich_url_from_tag_args {
  my ($url,$type,$args) = @_;
# Handle the default arg first  
  enrich_url_from_default_arg( $url, $type, $args->{_DEFAULT} );
# Handle further arguments     
  my @param = $type eq 'addi' ? get_addi_parameters($args) : @ROUTED_PARAMETERS;
  foreach my $name (@param) {
    next unless my $value = $args->{$name};
    for ($value) {
       s/^true$/X/;
       s/^false$/ /;
       };
    $url->query_param( $name, $value ); 
    }
  }
  

# Analyze the default argument of the TWiki tag
sub enrich_url_from_default_arg {
  my ($url,$type,$default_arg) = @_;
  my ($id,$path_ext);
  
  if ($type ~~ ['docu','code','']) {
    check_docu_key($default_arg);
    ($path_ext,$id) = ($type,$default_arg);
    }
  else {
   return unless $default_arg;
   ($path_ext,$id) =  $default_arg =~ /(.*?)\.(.*)/ ?
      ($1,$2) : ($default_arg,"");
    }  
  $url->path( $url->path . lc $path_ext );
  $url->query_param('id',$id) if $id;    
  }


# Returns an URI object for the connection to SAP, 
# Ready for adding path extension and further query params
sub get_url_from_dest {
  my $dest = uc ( shift || $defaultDest );
  my $domainAndPort = $sapDest{$dest} or die "Unknown system $dest\n";
  return URI->new("$scheme://$domainAndPort$sapPath");
  }
  

# Create URL for Topic names
# These topics should be general-purpose, 
# containing an %INCLSAPDOCU% or #INCLSAPCODE# tag, respectively
sub get_topic_url_for_link {  
  my $type = shift;  
# Map :
#  'docu' to TWiki.SapDocu
#  'code' to TWiki.SapCode
  my @topic = ('TWiki','Sap'.ucfirst lc $type);
  return URI->new( TWiki::Func::getScriptUrl(@topic,"view") );    
  }  


# Is the argument a valid docu key, like e.g. "RE.ZZ_REPORT"?
sub check_docu_key {
  my $arg = shift;
  $arg =~ /^(\w\w)\.([^.]+)/i or
    die "Illegal key for docu object: $arg\n";
  }


# The parameters of an INCLSAPADDI tag which are to be routed to SAP
sub get_addi_parameters {
  my $args = shift;
  my @param;
  for my $name (keys %$args) {
    next if $name eq '_DEFAULT' or $name eq 'dest';
    push @param, $name;
    }
  return @param;  
  }  


# Perform the actual HTTP request
sub get_request {
    
  my $url = shift;
     
  my $client = LWP::UserAgent->new;
  $client->timeout(10);
  $client->env_proxy;
  
  my $response = $client->get($url);
   
  if ($response->is_success) {
   return $response->decoded_content;
   }
  else { 
    die "HTTP request error: ".$response->status_line."\n";
    }
   
  }
  
  
# For code display: Preformats argument
sub preformat {
  my $content = shift;
  for ($$content) {
# Encode HTML-Tags and escape symbol
    s/\&/\&amp;/sg;
    s/</\&lt;/sg;
# Different style for ABAP comments
    s/^(\*.*)$/<span style="color:blue">$1<\/span>/sg;
# Wrap everything in a <PRE> ... </PRE>
    $_ = "<pre>\n$_\n</pre>";
    }  
  }

# Read the available SAP systems and the request path (once, early)
sub readSapDestinations {

# We constantly use the scheme http (dynamize here if necessary)
  $scheme = 'http';

  my $systemsParameter = TWiki::Func::getPluginPreferencesValue( "SYSTEMS" );

# First system in parameter =: default
  ($defaultDest) = $systemsParameter =~ /(\w+)/;   

# %sapDest: Hash containing system infos: domain and port
  %sapDest = $systemsParameter =~ /(\w+):([\w.]+:\d{4})\s*(?:,|$)\s*/g;

# Get request path for SAP system
  $sapPath = TWiki::Func::getPluginPreferencesValue( "PATH" )  || "/sap/bc/twiki/";

# Patch path with / if necessary
  $sapPath .= "/" unless $sapPath =~ /\/$/;
  $sapPath = "/" . $sapPath unless $sapPath =~ /^\//;
  
  }


# Produce an HTML-formatted error string
sub errorMsg {

  my $theMsg = shift;

  $theMsg =~ s/&/&amp;/g;
  $theMsg =~ s/</&lt;/g;

  $theMsg = '<pre style="color:red;font-weight:bold">' .
         $theMsg .
         '</pre>' ;
  return $theMsg;

  }


1;

# EOF