# $Id: Transforms.pm 488 2005-06-13 20:37:04Z nodine $

# Copyright (C) 2002-${YEAR} Freescale Semiconductor
# All rights reserved.

@Transforms::TRANSFORMS = qw(docutils.transforms.frontmatter.DocTitle
			     docutils.transforms.frontmatter.DocInfo
			     docutils.transforms.references.IndTargets
			     docutils.transforms.references.CitationReferences
			     docutils.transforms.misc.Pending
			     docutils.transforms.references.AutoFootnotes
			     docutils.transforms.references.FootnoteReferences
			     docutils.transforms.references.References
			     docutils.transforms.references.Unreferenced
			     docutils.transforms.universal.ScoopMessages
			     docutils.transforms.universal.Messages
			     docutils.transforms.universal.Decorations
			     );

package docutils::transforms::components;

=pod
=begin reST
=begin Usage
Defines for reStructuredText transforms
---------------------------------------
-D generator=<0|1>     Include a "Generated by" credit at the end of
                       the document (default is 1).
-D date=<0|1>          Include the date at the end of the document
                       (default is 0).
-D time=<0|1>          Include the date and time at the end of the
                       document (default is 1, overrides date if 1).
-D source_link=<0|1>   Include a "View document source" link (default
                       is 1).
-D source_url=<URL>    Use the supplied <URL> verbatim for a "View
                       document source" link; implies -D source_link=1.
-D keep_title_section  Keeps the section intact from which the document
                       title is taken.
=end Usage
=end reST
=cut

# Global variables:
#   ``@Transforms::TRANSFORMS``
#     Array of transform names in the order they will be applied.

use strict;

# Processes a docutils.transforms.components.Filter transform.
# Arguments: pending DOM, top-level DOM, parent DOM, details hash reference
sub Filter{
    my ($dom, $topdom, $parent, $details) = @_;

    if ($main::opt_w eq eval($details->{format}) ||
	$main::opt_w eq 'dom') {
	my $nodes = $details->{nodes};
	return new DOM($nodes->{tag}, %{$nodes->{attr}});
    }
    return;
}

package docutils::transforms::frontmatter;

use vars qw(%BIB_ELEMENTS);

BEGIN {
    my @bib_elements = qw(author authors organization address contact version
			  revision status date copyright dedication abstract);
    @BIB_ELEMENTS{@bib_elements} = (1) x @bib_elements;
}

# Processes a docutils.transforms.frontmatter.DocInfo transform.
# Processes field lists at the beginning of the DOM that are one of
# the docinfo types into a docinfo section.
# Arguments: top-level DOM
sub DocInfo {
    my ($dom) = @_;

    # Create a docinfo if needed
    my @field_lists = grep($_->{tag} eq 'field_list',
			   @{$dom->{content}});
    my %element_seen;
    if (@field_lists) {
	my $fl = $field_lists[0];
	my @content = @{$fl->{content}};
	# Modify the field list in situ
	$fl->{tag} = 'docinfo';
	@{$fl->{content}} = ();
	my $docinfo = $fl;
	my $field;
	my @predocinfo;# Things to be added to content list after docinfo
	    foreach $field (@content) {
		my $fn = $field->{content}[0];
		my $fb = $field->{content}[1];
		my $name = $fn->{content}[0]{text};
		my $origname = $name;
		$name =~ tr/A-Z/a-z/;
		my $tname = $name;
		substr($tname,0,1) =~ tr/ad/AD/;
		if ($BIB_ELEMENTS{$name}) {
		    $element_seen{$name}++;
		    if ($element_seen{$name} > 1 && $name =~ /abstract/) {
			$fb->append
			    (RST::system_message(2, $field->{source},
						 $field->{lineno},
						 qq(There can only be one "$tname" field.)));
			$docinfo->append($field);
		    }
		    elsif ($name =~ /^(dedication|abstract)$/) {
			my $topic = new DOM('topic', class=>$name);
			my $title = new DOM('title');
			$topic->append($title);
			$title->append(newPCDATA DOM($tname));
			$topic->append(@{$fb->{content}});
			push(@predocinfo, $topic);
		    }
		    elsif (@{$fb->{content}} < 1) {
			$fb->append
			    (RST::system_message(2, $field->{source},
						 $field->{lineno},
						 qq(Cannot extract empty bibliographic field "$origname".)));
			$docinfo->append($field);
		    }
		    elsif ($name eq 'authors') {
			my $bib = new DOM($name);
			my @contents = @{$fb->{content}[0]{content}};
			# There are three cases: bullet_lists,
			# multiple paragraphs, and string.
			if ((@{$fb->{content}} == 1 &&
			     ($fb->{content}[0]{tag} !~
			      /paragraph|bullet_list/ ||
			      $fb->{content}[0]{tag} eq 'bullet_list' &&
			      grep(@{$_->{content}} != 1 ||
				   $_->{content}[0]{tag} ne 'paragraph',
				   @{$fb->{content}[0]{content}}))
			     ) ||
			    (@{$fb->{content}} > 1 &&
			     grep($_->{tag} ne 'paragraph',
				  @{$fb->{content}}))
			    ) {
			    $fb->append
				(RST::system_message(2, $field->{source},
						     $field->{lineno},
						     qq(Bibliographic field "Authors" incompatible with extraction: it must contain either a single paragraph (with authors separated by one of ";,"), multiple paragraphs (one per author), or a bullet list with one paragraph (one author) per item.)));
			    $docinfo->append($field);
			}
			elsif (@{$fb->{content}} > 1) {
			    # Multiple paragraphs
			    foreach (@{$fb->{content}}) {
				my $author = new DOM('author');
				$bib->append($author);
				$author->append(@{$_->{content}});
			    }
			}
			elsif ($fb->{content}[0]{tag} eq 'bullet_list') {
			    my $bl = $fb->{content}[0];
			    foreach (@{$bl->{content}}) {
				my $author = new DOM('author');
				$bib->append($author);
				$author->append(@{$_->{content}[0]{content}});
			    }
			}
			else {
			    my $text;
			    $fb->Recurse(sub {
					     my ($dom) = @_;
					     $text .= $dom->{text}
					     if $dom->{tag} eq '#PCDATA';
					 }
					 );
			    my @authors = $text =~ /;/ ?
				split(/\s*;\s*/, $text) :
				split(/\s*,\s*/, $text);
			    foreach (@authors) {
				my $author = new DOM('author');
				$bib->append($author);
				$author->append(newPCDATA DOM($_));
			    }
			}
			if (@{$bib->{content}} == 1) {
			    $docinfo->append($bib->{content}[0]);
			}
			elsif (@{$bib->{content}} > 1) {
			    $docinfo->append($bib);
			}
		    }
		    elsif (@{$fb->{content}} > 1) {
			$fb->append
			    (RST::system_message(2, $field->{source},
						 $field->{lineno},
						 qq(Cannot extract compound bibliographic field "$origname".)));
			$docinfo->append($field);
		    }
		    elsif ($fb->{content}[0]{tag} ne 'paragraph') {
			$fb->append
			    (RST::system_message(2, $field->{source},
						 $field->{lineno},
						 qq(Cannot extract bibliographic field "$origname" containing anything other than a single paragraph.)));
			$docinfo->append($field);
		    }
		    else {
			my $bib = new DOM($name);
			%{$bib->{attr}} = (%RST::XML_SPACE)
			    if $name =~ /^address$/i;
			$docinfo->append($bib);
			my @contents = @{$fb->{content}[0]{content}};
			my $pcdata = $contents[0];
			$pcdata->{text} = $1
			    if defined $pcdata->{text} &&
			    $pcdata->{text} =~ /\$\w+:\s*(\S+?)(?:,v)?\s/;
			$bib->append(@contents);
		    }
		}
		else {
		    $docinfo->append($field);
		}
	    }

	# Anything before the docinfo that's not a title or subtitle
	# has to move after it.
	my $i;
	my $docinfo_seen = 0;
	my @new_content;  
	for ($i=0; $i<@{$dom->{content}}; $i++) {
	    my $c = $dom->{content}[$i];
	    if ($docinfo_seen || $c->{tag} =~ /^(sub)?title$/) {
		push @new_content, $c;
	    }
	    elsif ($c->{tag} eq 'docinfo') {
		$docinfo_seen = 1;
		push @new_content, $c, @predocinfo;
	    }
	    else {
		push @predocinfo, $c;
	    }
	}
	$dom->{content} = \@new_content;
    }
}

# Processes a docutils.transforms.frontmatter.DocTitle transform.
# Creates a document title if the top-level DOM has only one top-level
# section.  Creates a subtitle if a unique top-level section has a
# unique second-level section.
# Arguments: top-level DOM
sub DocTitle {
    my ($dom) = @_;
    # If the document has one section, coalesce it with the 'document'
    my @sections = grep($_->{tag} eq 'section', @{$dom->{content}});
    
    if (@sections == 1 && ($main::opt_D{keep_title_section} ||
			   !grep($_->{tag} !~
				 /section|comment|system_message/,
				 @{$dom->{content}}))) {
	my $sec = $sections[0];
	my @non_sections = grep($_->{tag} ne 'section',
				@{$dom->{content}});
	@{$dom->{attr}}{keys %{$sec->{attr}}} = values %{$sec->{attr}};
	if ($main::opt_D{keep_title_section}) {
	    my $title = $sec->{content}[0];
	    unshift(@{$dom->{content}}, $title);
	}
	else {
	    $dom->{content} = $sec->{content};
	    splice(@{$dom->{content}}, 1, 0, @non_sections);

	    # Check for a subtitle
	    my @sections = grep($_->{tag} eq 'section',
				@{$dom->{content}});
	    if (@sections == 1) {
		my $sec = $sections[0];
		my $title = shift(@{$sec->{content}});
		my @non_sections = grep($_->{tag} !~ /^(section|title)$/,
					@{$dom->{content}});
		unshift (@{$sec->{content}}, @non_sections);
		$title->{tag} = 'subtitle';
		$title->{attr} = $sec->{attr};
		@{$dom->{content}} = grep($_->{tag} eq 'title',
					    @{$dom->{content}});
		push(@{$dom->{content}}, $title, @{$sec->{content}});
	    }
	}
    }
}

package docutils::transforms::misc;

# This package contains routines for transforms of DOM trees

# Processes a docutils.transforms.misc.Pending transform.
# Traverses the DOM tree looking for Pending nodes and applies
# whatever internal transform was specified for them.
# Arguments: top-level DOM
sub Pending {
    my ($topdom) = @_;

    # Handle pending transformations
    $topdom->Reshape
	(sub {
	     my($dom, $parent) = @_;
	     if ($dom->{tag} eq 'pending') {
		 my $transform = $dom->{internal}{'.transform'};
		 (my $t = $transform) =~ s/\./::/g;
		 return RST::system_message(4, $dom->{source},
					    $dom->{lineno},
					    qq(No transform code found for "$transform".))
		     unless defined &$t;
		 my $details = $dom->{internal}{'.details'};
		 no strict 'refs';
		 print STDERR "Debug: Transform $transform\n" if $main::opt_d;
		 return &$t($dom, $topdom, $parent, $details);
	     }
	     return $dom;
	 });
}

package docutils::transforms::parts;

# This package contains routines for transforms of DOM trees

# Processes a docutils.transforms.parts.Contents transform.
# Arguments: pending DOM, top-level DOM, parent DOM, details hash reference
sub Contents {
    my ($dom, $topdom, $parent, $details) = @_;

    my $backlinks =
	defined $details->{backlinks} ? $details->{backlinks} : '';
    # First we compile the table of contents
    my $topic = new DOM('topic', class=>'contents');
    my $titledom = $details->{title};
    if ($parent->{tag} eq 'document' && !$titledom) {
	$titledom = new DOM('title');
	$titledom->append(newPCDATA DOM('Contents'));
    }
    my $titlename = $titledom ? $titledom->{content}[0]{text} :
	'Contents';
    my $name = RST::NormalizeName($titlename);
    my @errs;
    if (defined $RST::TARGET_NAME{target}{$name}) {
	$topic->{attr}{id} = RST::Id();
	$topic->append($titledom) if $titledom && $backlinks =~ /top|none/i;
    }
    else {
	$topic->append($titledom) if $titledom;
	$topic->{attr}{name} = $name;
	$topic->{attr}{id} = RST::NormalizeId($titlename);
	my $err = RST::RegisterName($topic, $dom->{source}, $dom->{lineno});
	push (@errs, $err) if $err ne '';
    }
    my $contid = $topic->{attr}{id};

    my $bl = new DOM('bullet_list');
    my $depth = 0; # Used in closure of sub
    my @list = ($bl); # Used in closure of sub
    my $start = (defined $details->{local}) ? $parent : $topdom;
    $start->Recurse
	(sub {
	     my($dom, $parent, $when) = @_;
	     if ($dom->{tag} eq 'section' && defined $parent) {
		 $depth-- if $when eq 'post';
		 if (! defined $details->{depth} ||
		     $depth < $details->{depth}) {
		     my $bl = $list[-1];
		     my $li; 
		     if ($when eq 'pre') {
			 my $id = RST::Id();
			 $li = new DOM('list_item'); #, id=>$id);
			 $bl->append($li);
			 if ($backlinks !~ /none/i) {
			     $dom->{content}[0]{attr}{refid} =
				 ($backlinks =~ /top/i) ? $contid : $id;
			 }
			 my $para = new DOM('paragraph');
			 $li->append($para);
			 my $ref = new DOM('reference', id=>$id,
					   refid=>$dom->{attr}{id});
			 $para->append($ref);
			 my @contents;  # Used in the closure of the sub
			 $dom->{content}[0]->Recurse
			     (sub {
				  my ($dom, $parent, $when) = @_;
				  my $tag = $dom->{tag};
				  if ($tag =~ /^(?:title|(footnote|citation)_reference|interpreted|problematic|reference|target)$/) {
				      # Ignore
				  }
				  elsif ($tag =~ /image/) {
				      push(@contents,
					   newPCDATA DOM($dom->{attr}{alt}))
					  if defined $dom->{attr}{alt} &&
					  $when eq 'pre';
				  }
				  else {
				      if ($when eq 'pre') {
					  # Don't recurse
					  push(@contents, $dom);
					  return 1;
				      }
				  }
				  return 0;
			      }
			      , $parent, 'both');
			 $ref->append(@contents);
			 $list[0]{attr}{class} = 'auto-toc'
			     if ($dom->{content}[0]{content}[0]{tag}
				 eq 'generated');
		     }

		     # Check to see if I have any nested sections
		     if (grep($_->{tag} eq 'section',@{$dom->{content}})
			 && (! defined $details->{depth} ||
			     $depth < $details->{depth}-1)) {
			 if ($when eq 'pre') {
			     my $new_bl = new DOM('bullet_list');
			     $li->append($new_bl);
			     push(@list, $new_bl);
			 }
			 else {
			     pop(@list);
			 }
		     }
		 }
		 $depth++ if $when eq 'pre';
	     }
	     return 0;
	 }
	 , undef, 'both') ;
    $topic->append($bl) if @{$bl->{content}};
    return unless grep($_->{tag} eq 'bullet_list', @{$topic->{content}});
    return $topic, @errs;
}

# Processes a docutils.transforms.parts.Sectnum transform.
# Auto-numbers the sections in the document.
# Arguments: pending DOM, top-level DOM, parent DOM, details hash reference
sub Sectnum {
    my ($dom, $topdom, $parent, $details) = @_;

    # First process the table of contents topic if it exists
    my ($toc) = grep($_->{tag} eq 'topic' &&
		     $_->{attr}{class} eq 'contents',
		     @{$topdom->{content}});
    my @list; # Used in closure of sub
    if (defined $toc) {
	$toc->Recurse
	    (sub {
		 my($dom, $parent, $when) = @_;
		 if ($dom->{tag} eq 'bullet_list') {
		     if ($when eq 'pre') {
			 push(@list, 0);
			 $dom->{attr}{class} = 'auto-toc'
			     if (! defined $details->{depth} ||
				 @list <= $details->{depth});
		     }
		     else { pop(@list) };
		 }
		 elsif ($dom->{tag} eq 'list_item' && $when eq 'pre') {
		     $list[-1]++;
		 }
		 elsif ($dom->{tag} eq 'reference' && $when eq 'pre'
			&& (! defined $details->{depth} ||
			    @list <= $details->{depth})) {
		     my $gen = new DOM('generated', class=>'sectnum');
		     $gen->append(newPCDATA DOM(join('.',@list)
						. ('\u00a0'x3)));
#						. ("\302\240"x3)));
		     unshift(@{$dom->{content}}, $gen);
		 }
		 return 0;
	     }
	     , $topdom, 'both') ;
    }

    # Next process the sections recursively
    @list = (0);
    $topdom->Recurse
	(sub {
	     my($dom, $parent, $when) = @_;
	     if ($dom->{tag} eq 'section') {
		 if ($when eq 'pre') {
		     if (! defined $details->{depth} ||
			 @list <= $details->{depth}) {
			 my $title = $dom->{content}[0];
			 $title->{attr}{auto} = 1;
			 $list[-1]++;
			 my $gen = new DOM('generated', class=>'sectnum');
			 $gen->append(newPCDATA DOM(join('.',@list)
						    . ('\u00a0'x3)));
#						. ("\302\240"x3)));
			 unshift(@{$title->{content}}, $gen);
		     }
		     push(@list, 0);
		 }
		 else { pop(@list); }
	     }
	     return 0;
	 }
	 , undef, 'both') ;
    
    return;
}

package docutils::transforms::references;

# This package contains routines for transforms of DOM trees

# Static global variables
use vars qw($NEXT_SYMBOL_FOOTNOTE @FOOTNOTE_SYMBOLS);

# Run-time global variables
use vars qw($AUTO_FOOTNOTE_REF $LAST_AUTO_FOOTNOTE @AUTO_FOOTNOTES);

BEGIN {
    @FOOTNOTE_SYMBOLS = ("*", "\342\200\240", "\342\200\241", "\302\247",
			 "\302\266", "#", "\342\231\240", "\342\231\245",
			 "\342\231\246", "\342\231\243");
    $NEXT_SYMBOL_FOOTNOTE = 0;
}

# Processes a docutils.transforms.references.AutoFootnotes transform.
# Computes numbers for autonumbered footnotes.
# Arguments: top-level DOM
sub AutoFootnotes {
    my ($dom) = @_;

    # Compute numbers for autonumbered footnotes
    $dom->Recurse
	(sub {
	     my($dom, $parent) = @_;
	     my $tag = $dom->{tag};
	     if ($tag eq 'footnote') {
		 if ($dom->{attr}{auto}) {
		     my $label = new DOM('label');
		     $dom->prepend($label);
		     if ($dom->{attr}{auto} eq '1') {
			 while (defined $RST::REFERENCE_DOM{$tag}
				{++$LAST_AUTO_FOOTNOTE}) { };
			 if (! defined $dom->{attr}{name} &&
			     ! defined $dom->{attr}{dupname}) {
			     push(@AUTO_FOOTNOTES, $dom);
			     $dom->{attr}{name} = $LAST_AUTO_FOOTNOTE;
			     RST::RegisterName($dom, $dom->{source},
					       $dom->{lineno});
			 }
			 $label->append(newPCDATA
					DOM($LAST_AUTO_FOOTNOTE));
		     }
		     else {
			 push(@AUTO_FOOTNOTES, $dom);
			 my $multiplier =
			     int($NEXT_SYMBOL_FOOTNOTE/@FOOTNOTE_SYMBOLS) + 1;
			 my $index =
			     $NEXT_SYMBOL_FOOTNOTE % @FOOTNOTE_SYMBOLS;
			 my $name =
			     ($FOOTNOTE_SYMBOLS[$index]) x $multiplier;
			 $label->append(newPCDATA DOM($name));
			 $NEXT_SYMBOL_FOOTNOTE++;
		     }
		     $RST::REFERENCE_DOM{$tag}{$dom->{attr}{name}} = $dom
			 if defined $dom->{attr}{name};
		     $RST::REFERENCE_DOM{$tag}{$dom->{attr}{id}} = $dom;
		 }
	     }
	     return 0;
	 }
	 , undef, 'pre');
}

# Processes a docutils.transforms.references.IndTargets transform.
# Links indirect targets to their eventual destination.
# Arguments: top-level DOM
sub IndTargets {
    my ($dom) = @_;

    # Process indirect targets
    $dom->Recurse
	(sub {
	     my($dom, $parent) = @_;
	     my $tag = $dom->{tag};
	     if ($tag eq 'target') {
		 if (! grep(/^ref(name|uri|id)$/, keys %{$dom->{attr}})) {
		     # This was just a bare target.
		     # Tie it to what's next if it is a target
		     my $me;
		     for ($me=0; $me<@{$parent->{content}}; $me++) {
			 last if $parent->{content}[$me] == $dom;
		     }
		     my $i;
		     for ($i=$me+1; $i<@{$parent->{content}}; $i++) {
			 my $next = $parent->{content}[$i];
			 my @refkeys = grep(/^ref(name|uri)$/,
					    keys %{$next->{attr}});
			 last if ($next->{tag} ne 'target' ||
				  (defined $next->{attr}{name} &&
				   @refkeys == 0));
			 next unless defined $next->{attr}{name} ||
			     @refkeys;
			 if (defined $next->{attr}{name}) {
			     # We'll resolve this below
			     $dom->{attr}{refname} = $next->{attr}{name};
			 }
			 else {
			     my $ref = $refkeys[0];
			     $dom->{attr}{$ref} = $next->{attr}{$ref};
			 }
			 last;
		     }
		 }
		 
		 if (defined $dom->{attr}{refname}) {
		     my $target = $dom;
		     $target->{type} = "Indirect"
			 unless defined $target->{type};
		     while (defined $target->{attr}{refname}) {
			 my $name = $target->{attr}{refname};
			 my @targets = grep(defined $_->{attr}{name} &&
					    $_->{attr}{name} eq $name,
					    @{$RST::ALL_TARGET_NAMES{$name}});
			 $target = $targets[0];
			 $target->{referenced} = 1 if defined $target;
		     }
		     if (defined $target->{attr}{refuri}) {
			 delete $dom->{attr}{refname};
			 $dom->{attr}{refuri} = $target->{attr}{refuri};
		     }
		     elsif (defined $target->{attr}{refid}) {
			 delete $dom->{attr}{refname};
			 $dom->{attr}{refid} = $target->{attr}{refid};
		     }
		     elsif (defined $target->{attr}{id}) {
			 delete $dom->{attr}{refname};
			 $dom->{attr}{refid} = $target->{attr}{id};
		     }
		 }
	     }
	     return 0;
	 }
	 , undef, 'pre');
}

# Processes a docutils.transforms.references.CitationReferences transform.
# Links citation references to their targets.
# Arguments: top-level DOM
sub CitationReferences {
    my ($dom) = @_;

    # Link references to their definitions if they exist
    $dom->Reshape
	(\&CitationReferences_,
	 , undef, 'pre');
}

# INTERNAL ROUTINE.
# Used recursively by CitationReferences.
# Arguments: DOM object, parent
sub CitationReferences_ {
    my($dom, $parent) = @_;
    my $tag = $dom->{tag};
    if ($tag =~ /^(?:(citation|substitution)_reference)$/ &&
	$parent->{tag} ne 'substitution_definition') {
	my $what = $1 eq 'citation' ? $1 :
	    'substitution_definition';
	my $name = main::FirstDefined($dom->{attr}{name},
				      $dom->{attr}{refname});
	my $target = $RST::REFERENCE_DOM{$what}{$name};
	if (! defined $target) {
	    my ($prob, $refid, $id) =
		RST::problematic($dom->{lit});
	    my $emsg = $what eq 'citation' ?
		'Unknown target name' :
		'Undefined substitution referenced';
	    return
		(RST::system_message
		 (3, $dom->{source}, $dom->{lineno},
		  qq($emsg: "$name".), '', id=>$refid, backrefs=>$id),
		 $prob);
	}
	if ($tag eq 'substitution_reference') {
	    my @content= @{$target->{content}};
	    my $i;
	    for ($i=0; $i<@content; $i++) {
		splice(@content, $i, 1,
		       CitationReferences_($content[$i], $parent))
		    if $content[$i]{tag} eq 'substitution_reference';
	    }
	    return @content;
	}
	return @{$target->{content}}
	if $tag eq 'substitution_reference';
	delete $dom->{attr}{refname};
	$dom->{attr}{refid} = $target->{attr}{id};
	$target->{attr}{backrefs} .=
	    ($target->{attr}{backrefs} ? ' ' : '') . $dom->{attr}{id};
    }
    return $dom;
}

# Processes a docutils.transforms.references.FootnoteReferences transform.
# Links footnote references to their targets.
# Arguments: top-level DOM
sub FootnoteReferences {
    my ($dom) = @_;

    # Link references to their definitions if they exist
    $dom->Reshape
	(sub {
	     my($dom) = @_;
	     my $tag = $dom->{tag};
	     if ($tag eq 'footnote_reference' && !$dom->{resolved}) {
		 my $name = main::FirstDefined($dom->{attr}{name},
					       $dom->{attr}{refname});
		 my $footnote = defined $name ?
		     $RST::REFERENCE_DOM{footnote}{$name} :
		     $AUTO_FOOTNOTES[$AUTO_FOOTNOTE_REF++];
		 if (! defined $footnote) {
		     my ($prob, $refid, $id) =
			 RST::problematic($dom->{lit});
		     return
			 ($prob, RST::system_message
			  (3, $dom->{source}, $dom->{lineno},
			   (defined $name ? qq(Unknown target name: "$name".):
			    qq(Too many autonumbered footnote references: only ${\scalar(@AUTO_FOOTNOTES)} corresponding footnotes available.)),
			   '', id=>$refid, backrefs=>$id),
			  );
		 }
		 if ($dom->{attr}{auto}) {
		     my $name = $footnote->{content}[0]{content}[0]{text};
		     $dom->append(newPCDATA DOM($name));
		 }
		 delete $dom->{attr}{refname};
		 $dom->{attr}{refid} = $footnote->{attr}{id};
		 $footnote->{attr}{backrefs} .=
		     ($footnote->{attr}{backrefs} ? ' ' : '')
		     . $dom->{attr}{id};
		 $dom->{resolved} = 1;
	     }
	     return $dom;
	 }
	 , undef, 'pre');
}

# Processes a docutils.transforms.references.References transform.
# Counts anonymous references, links references to their
# destinations, produces error messages if the number of anonymous
# references is insufficient.
# Arguments: top-level DOM
sub References {
    my ($dom) = @_;

    my $anonymous_refs;
    # Count how many anonymous references we have
    $dom->Recurse
	(sub {
	     my($dom) = @_;
	     $anonymous_refs++
		 if ($dom->{tag} eq 'reference' && $dom->{attr}{anonymous});
	     return 0;
	 }
	 , undef, 'pre');
    my $last_anonymous_target = 0;
    my $anonymous_mismatch_id;
    my @anonymous_mismatch_refids;
    # Link references to their definitions if they exist
    $dom->Reshape
	(sub {
	     my($dom) = @_;
	     my $tag = $dom->{tag};
	     if ($tag eq 'reference' && ! defined $dom->{attr}{refuri} &&
		 ! defined $dom->{attr}{refid}) {
		 my $target;
		 my $name = $dom->{attr}{refname};
		 if (defined $name) {
		     my @targets = @{$RST::TARGET_NAME{target}{$name}}
		     if defined $RST::TARGET_NAME{target}{$name};
		     $target = $targets[0] if @targets == 1;
		     if (! defined $target &&
			 ! defined $RST::ALL_TARGET_IDS{$name}[0]) {
			 my ($prob, $refid, $id) =
			     RST::problematic($dom->{lit});
			 return
			     (RST::system_message
			      (3, $dom->{source}, $dom->{lineno},
			       qq(Unknown target name: "$name".),
			       '', id=>$refid, backrefs=>$id),
			      $prob);
		     }
		 }
		 elsif ($dom->{attr}{anonymous}) {
		     if ($anonymous_refs > @RST::ANONYMOUS_TARGETS) {
			 $anonymous_mismatch_id = RST::Id()
			     if ! defined $anonymous_mismatch_id;
			 my ($prob, $refid, $id) =
			     RST::problematic($dom->{lit},
					      $anonymous_mismatch_id);
			 push(@anonymous_mismatch_refids, $id);
			 return $prob;
		     }
		     $target =
			 $RST::ANONYMOUS_TARGETS[$last_anonymous_target++];
		 }
		 while (defined $target) {
		     $target->{referenced} = 1;
		     if (defined $target->{attr}{refuri}) {
			 $target->{type} = "External"
			     unless defined $target->{type};
			 delete $dom->{attr}{refname};
			 $dom->{attr}{refuri} = $target->{attr}{refuri};
		     }
		     elsif (defined $target->{attr}{refid}) {
			 $target->{type} = "Internal"
			     unless defined $target->{type};
			 my @targets = @{$RST::ALL_TARGET_IDS
					 {$target->{attr}{refid}}};
			 $target = $targets[0];
			 next;
		     }
		     elsif (defined $target->{attr}{refname}) {
			 $target->{type} = "Indirect"
			     unless defined $target->{type};
			 my ($prob, $refid, $id) =
			     RST::problematic($dom->{lit});
			 my $tname = defined $target->{attr}{name} ?
			     qq("$target->{attr}{name}" ) : "";
			 return (RST::system_message
				 (2, $target->{source}, $target->{lineno},
				  qq(Indirect hyperlink target $tname(id="$target->{attr}{id}") refers to target "$target->{attr}{refname}", which does not exist.),
				  '', id=>$refid, backrefs=>$id),
				 $prob);
		     }
		     elsif (defined $target->{attr}{id}) {
			 delete $dom->{attr}{refname};
			 $dom->{attr}{refid} = $target->{attr}{id};
		     }
		     undef $target;
		 }
	     }
	     return $dom;
	 }
	 , undef, 'pre');

    # Produce messages if there aren't enough anonymous hyperlink targets
    if (defined $anonymous_mismatch_id) {
	my $sm = RST::system_message
	    (3, $dom->{attr}{source}, undef,
	     qq(Anonymous hyperlink mismatch: $anonymous_refs references but ${\scalar(@RST::ANONYMOUS_TARGETS)} targets.\nSee "backrefs" attribute for IDs.),
	     '', id=>$anonymous_mismatch_id,
	     backrefs=>join(' ',@anonymous_mismatch_refids));
	delete $sm->{attr}{line};
	$dom->append($sm);
    }
}

# Processes a docutils.transforms.references.Unreferenced transform.
# Produces messages for unreferenced targets.
# Arguments: top-level DOM
sub Unreferenced {
    my ($dom) = @_;

    # Produce messages for unreferenced targets
    $dom->Reshape
	(sub {
	     my($dom) = @_;
	     my $tag = $dom->{tag};
	     if ($tag eq 'target' && ! $dom->{referenced} &&
		 ! $dom->{attr}{anonymous}) {
		 my $name = $dom->{attr}{name};
		 my $type = main::FirstDefined($dom->{type},
					       defined $dom->{refuri} ?
					       "External" : "Internal");
		 my $id = defined $name ? qq("$name") :
		     qq(id="$dom->{attr}{id}");
		 return ($dom,
			 RST::system_message(1, $dom->{source},
					     $dom->{lineno},
					     qq($type hyperlink target $id is not referenced.)));
	     }
	     return $dom;
	 }
	 , undef, 'pre');
}

# Processes a docutils.transforms.references.TargetNotes transform.
# Constructs a list of external references and creates footnotes
# pointing to them.
# Arguments: pending DOM, top-level DOM, parent DOM, details hash reference
sub TargetNotes {
    my ($dom, $topdom, $parent, $details) = @_;

    my @targets; # Used in closure of sub
    # Construct the list of external references.
    $topdom->Recurse
	(sub {
	     my($dom, $parent) = @_;
	     my $tag = $dom->{tag};
	     push (@targets, $dom)
		 if $tag eq 'target' && defined $dom->{attr}{refuri};
	     return 0;
	 }) ;

    # Create the footnotes
    my @doms;
    my %footnotes;
    foreach (@targets) {
	my $id = RST::Id();
	$footnotes{$_->{attr}{name}} = $id;
	my $dom = new DOM('footnote', auto=>1, id=>$id,
			  name=>"target_note: $id");
	my $para = new DOM('paragraph');
	$dom->append($para);
	my $ref = new DOM('reference', refuri=>$_->{attr}{refuri});
	$para->append($ref);
	$ref->append(newPCDATA DOM($_->{attr}{refuri}));
	push(@doms, $dom);
    }

    # Insert the footnote references
    $topdom->Reshape
	(sub {
	     my($dom, $parent) = @_;
	     my $tag = $dom->{tag};
	     if ($tag eq 'reference' &&
		 defined $dom->{attr}{refname} &&
		 defined $footnotes{$dom->{attr}{refname}}) {
		 my $refname = $footnotes{$dom->{attr}{refname}};
		 my $fr = new DOM('footnote_reference', auto=>1,
				  id=>RST::Id(),
				  refname=>"target_note: $refname");
		 return ($dom, $fr);
	     }
	     return $dom;
	 }) ;
    
    return @doms;
}

package docutils::transforms::universal;

# This package contains routines for transforms of DOM trees

# Processes a docutils.transforms.universal.Decorations transform.
# Adds the "View docuemnt source", "Generated on" and "Generated by"
# decorations to the end of the document.
# Arguments: top-level DOM
sub Decorations {
    my ($topdom) = @_;

    my $para = new DOM('paragraph');
    my $source_link = main::FirstDefined($main::opt_D{source_link}, 1);
    if ($source_link) {
	my $source_url = main::FirstDefined($main::opt_D{source_url},
					    $topdom->{attr}{source});
	my $ref = new DOM('reference', refuri=>$source_url);
	$ref->append(newPCDATA DOM('View document source'));
	$para->append($ref);
	$para->append(newPCDATA DOM(".\n"));	
    }
    my $time = main::FirstDefined($main::opt_D{time}, 1);
    my $date = main::FirstDefined($main::opt_D{date}, 0);
    if ($date || $time) {
	my $format = "%Y/%m/%d" . ($time ? " %H:%M:%S %Z" : "");
	use POSIX;
	my $date = POSIX::strftime($format, localtime);
	$para->append(newPCDATA DOM("Generated on: $date.\n"));
    }
    my $generator = main::FirstDefined($main::opt_D{generator}, 1);
    if ($generator) {
	my $ref = new DOM('reference', refuri=>
			  'http://docutils.sourceforge.net/rst.html');
	$ref->append(newPCDATA DOM("reStructuredText"));
	$para->append(newPCDATA DOM("Generated by $main::GENERATOR_NAME from "),
		      $ref,
		      newPCDATA DOM(" source.\n"));
    }

    if (@{$para->{content}}) {
	my $dec = new DOM('decoration');
	my $footer = new DOM('footer');
	$dec->append($footer);
	$footer->append($para);
	# Decoration needs to be appended before the document model
	# starts, i.e., after the latest of title, subtitle, or docinfo.
	my $i;
	for ($i=0; $i<@{$topdom->{content}}; $i++) {
	    if ($topdom->{content}[$i]{tag} !~ /title|docinfo/) {
		splice(@{$topdom->{content}}, $i, 0, $dec);
		last;
	    }
	}
    }
}

# Processes a docutils.transforms.universal.Messages transform.
# Moves system messages at the end into "Docutils System Messages" section.
# Arguments: top-level DOM
sub Messages {
    my ($dom) = @_;

    # Move system messages at the end to a section
    my @SYSTEM_MESSAGES;
    $dom->Reshape
	(sub {
	     my($dom) = @_;
	     push (@SYSTEM_MESSAGES, $dom)
		 if ($dom->{tag} eq 'system_message' &&
		     $dom->{attr}{level} >= $main::opt_D{report});
	     return $dom->{tag} ne 'system_message' ? ($dom) : ();
	 }
	 );

   if (@SYSTEM_MESSAGES > 0) {
       my $errsec = new DOM('section', class=>'system-messages');
       $dom->append($errsec);
       my $title = new DOM('title');
       $errsec->append($title);
       $title->append(newPCDATA DOM('Docutils System Messages'));
       $errsec->append(@SYSTEM_MESSAGES);
    }
}

# Processes a docutils.transforms.universal.ScoopMessages transform.
# Moves system messages from anywhere in the DOM tree to the end of
# the top-level DOM.
# Arguments: top-level DOM
sub ScoopMessages {
    my ($dom) = @_;

    # Move system messages into end of top dom's contents
    my @SYSTEM_MESSAGES;
    $dom->Reshape
	(sub {
	     my($dom, $parent) = @_;
	     if ($dom->{tag} eq 'system_message') {
		 if (defined $dom->{attr}{id}) {
		     push(@SYSTEM_MESSAGES, $dom);
		     return;
		 }
	     }
	     return $dom;
	 });
    $dom->append(@SYSTEM_MESSAGES);
}

1;
