<!--
    OpenOffice.org XML to TWiki markup 

    Use this stylesheet to convert the 'content.xml' from an OpenOffice.org
    or StarOffice Writer document into a TWiki page.

    This stylesheet is for OpenDocument Text (.odt)

    http://www.openoffice.org
    http://www.twiki.org

    Maintainer: Wade Turland <wade.turland@optus.net.au>
    Based on soffice2html.pl by Steve Slaven
    and      OOo_TWiki.xslt by Brad Dixon (twiki.20.bdixon@spamgourmet.com)
-->
<xsl:stylesheet version="1.0"
    xmlns:math="http://www.w3.org/1998/Math/MathML"
    xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:xlink="http://www.w3.org/1999/xlink"
    xmlns:xsd="http://www.w3.org/2001/XMLSchema"
    xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
    xmlns:dom="http://www.w3.org/2001/xml-events"
    xmlns:xforms="http://www.w3.org/2002/xforms"
    xmlns:dc="http://purl.org/dc/elements/1.1/"
    xmlns:chart="urn:oasis:names:tc:opendocument:xmlns:chart:1.0"
    xmlns:number="urn:oasis:names:tc:opendocument:xmlns:datastyle:1.0"
    xmlns:dr3d="urn:oasis:names:tc:opendocument:xmlns:dr3d:1.0"
    xmlns:draw="urn:oasis:names:tc:opendocument:xmlns:drawing:1.0"
    xmlns:form="urn:oasis:names:tc:opendocument:xmlns:form:1.0"
    xmlns:meta="urn:oasis:names:tc:opendocument:xmlns:meta:1.0"
    xmlns:office="urn:oasis:names:tc:opendocument:xmlns:office:1.0"
    xmlns:script="urn:oasis:names:tc:opendocument:xmlns:script:1.0"
    xmlns:style="urn:oasis:names:tc:opendocument:xmlns:style:1.0"
    xmlns:svg="urn:oasis:names:tc:opendocument:xmlns:svg-compatible:1.0"
    xmlns:table="urn:oasis:names:tc:opendocument:xmlns:table:1.0"
    xmlns:text="urn:oasis:names:tc:opendocument:xmlns:text:1.0"
    xmlns:fo="urn:oasis:names:tc:opendocument:xmlns:xsl-fo-compatible:1.0"
    xmlns:oooc="http://openoffice.org/2004/calc"
    xmlns:ooo="http://openoffice.org/2004/office"
    xmlns:ooow="http://openoffice.org/2004/writer"
    >

<!-- This comes from soffice2html.pl but TWiki doesn't allow subdirs for images
<xsl:param name="image-prefix" select="'image/'"/ -->
<xsl:param name="image-prefix" select="'%ATTACHURLPATH%/'"/>
<xsl:param name="style-prefix" select="'soffice'"/>
<xsl:param name="image-extension" select="'png'"/>
<!-- Brad Dixon uses dpi of 105: "using 96 dpi makes everything feel 'small'" -->
<!-- OpenOffice uses dpi of 96 -->
<xsl:param name="dpi" select="'96'"/>

    <xsl:output method="html"/>
    <xsl:strip-space elements="*"/>
    <!-- Catch the non-content document sections -->
    <xsl:template match="/XML"/>
    <xsl:template match="/office:document/office:meta"/>
    <xsl:template match="/office:document/office:settings"/>
    <xsl:template match="/office:document/office:document-styles"/>
    <xsl:template match="/office:document/office:font-decls"/>
    <xsl:template match="/office:document/office:styles"/>
    <xsl:template match="/office:document/office:master-styles"/>

    <!-- Formats the text sections according to the style name -->
    <xsl:template name="style-font">
        <xsl:param name="style"/>

        <!-- get the style attributes from the style elements -->
        <xsl:variable name="font-style" select="//office:automatic-styles/style:style[@style:name=$style]/style:text-properties/@fo:font-style"/>
        <xsl:variable name="parent-style" select="//office:automatic-styles/style:style[@style:name=$style]/@style:parent-style-name"/>
        <xsl:variable name="font-weight" select="//office:automatic-styles/style:style[@style:name=$style]/style:text-properties/@fo:font-weight"/>

        <!-- Debug styles - ->
        <xsl:text>[</xsl:text>
        <xsl:value-of select="$font-weight"/><xsl:text>, </xsl:text>
        <xsl:value-of select="$font-style"/><xsl:text>, </xsl:text>
        <xsl:text>]</xsl:text>
        <!- - Debug styles -->
        <xsl:choose>
            <xsl:when test="$style='Table_20_Heading' or $parent-style='Table_20_Heading'">*</xsl:when>
            <xsl:when test="$style='Caption' or $parent-style='Caption'">_</xsl:when>
            <xsl:when test="$style='Source_20_Text'">=</xsl:when>
            <xsl:when test="$font-weight='bold' and $parent-style='Source_20_Text'">==</xsl:when>
            <xsl:when test="$font-weight='bold' and not($font-style='italic')">*</xsl:when>
            <xsl:when test="not($font-weight='bold') and $font-style='italic'">_</xsl:when>
            <xsl:when test="$font-weight='bold' and $font-style='italic'">__</xsl:when>
        </xsl:choose>
    </xsl:template>

    <!-- Text paragraphs -->
    <xsl:template match="//text:p">
        <xsl:variable name="cur-style-name">
            <xsl:value-of select="@text:style-name"/>
        </xsl:variable>
        <xsl:variable name="alignment" select="//office:automatic-styles/style:style[@style:name=$cur-style-name]/style:paragraph-properties/@fo:text-align"/>
        <xsl:variable name="text">
            <xsl:apply-templates/>
        </xsl:variable>
        <xsl:if test="$text != ''">
            <xsl:choose>
                <xsl:when test="$alignment='center' and not(ancestor::table:table)">
                    <xsl:call-template name="style-font">
                        <xsl:with-param name="style">
                            <xsl:value-of select="$cur-style-name"/>
                        </xsl:with-param>
                    </xsl:call-template>
                    <xsl:element name="center">
                        <xsl:apply-templates/>
                    </xsl:element>
                    <xsl:call-template name="style-font">
                        <xsl:with-param name="style">
                            <xsl:value-of select="$cur-style-name"/>
                        </xsl:with-param>
                    </xsl:call-template>
                </xsl:when>
                <xsl:when test="$alignment='end' and not(ancestor::table:table)">
                    <xsl:call-template name="style-font">
                        <xsl:with-param name="style">
                            <xsl:value-of select="$cur-style-name"/>
                        </xsl:with-param>
                    </xsl:call-template>
                    <xsl:element name="p">
                        <xsl:attribute name="align">right</xsl:attribute>
                        <xsl:apply-templates/>
                    </xsl:element>
                    <xsl:call-template name="style-font">
                        <xsl:with-param name="style">
                            <xsl:value-of select="$cur-style-name"/>
                        </xsl:with-param>
                    </xsl:call-template>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:call-template name="style-font">
                        <xsl:with-param name="style">
                            <xsl:value-of select="$cur-style-name"/>
                        </xsl:with-param>
                    </xsl:call-template>
                    <xsl:apply-templates/>
                    <xsl:call-template name="style-font">
                        <xsl:with-param name="style">
                            <xsl:value-of select="$cur-style-name"/>
                        </xsl:with-param>
                    </xsl:call-template>
                </xsl:otherwise>
            </xsl:choose>
            <!-- these are all newline rules -->
            <xsl:choose>
                <!-- we shouldn't add a newline for elements inside a table -->
                <xsl:when test="ancestor::table:table-cell and not(following-sibling::text:p)"/>
                <!-- but we do want a <br> between two text:p in a cell -->
                <xsl:when test="ancestor::table:table-cell and following-sibling::text:p">
                    <xsl:text disable-output-escaping="yes"> %BR% </xsl:text>
                </xsl:when>
                <!-- and we want a <br> between two text:p in a list-item -->
                <xsl:when test="ancestor::text:list-item and following-sibling::text:p">
                    <!--xsl:element name="br"/-->
                    <xsl:text disable-output-escaping="yes"> %BR%
</xsl:text>
                    <xsl:call-template name="expand-chars">
                        <xsl:with-param name="iterations">
                            <xsl:value-of select="count(ancestor::text:list)"/>
                        </xsl:with-param>
                        <xsl:with-param name="dup-chars">
                            <xsl:text disable-output-escaping="yes">	</xsl:text>
                        </xsl:with-param>
                    </xsl:call-template>
                </xsl:when>
                <!-- we do want a single newline at the end of a list item -->
                <xsl:when test="ancestor::text:list-item and following::text:list-item">
                    <xsl:text disable-output-escaping="yes">
</xsl:text>
                </xsl:when>
                <!-- and double newline at the end of the list-->
                <xsl:when test="ancestor::text:list-item">
                    <xsl:text disable-output-escaping="yes">

</xsl:text>
                </xsl:when>
                <xsl:otherwise>
                    <!-- normal paragraph ending -->
                    <xsl:text disable-output-escaping="yes">

</xsl:text>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:if>
        <!-- special cases: we still want to apply even if there is no text -->
        <xsl:if test="$text='' and descendant::draw:image">
            <xsl:apply-templates/>
        </xsl:if>
    </xsl:template>

    <!-- text spans -->
    <xsl:template match="//text:span">
        <xsl:variable name="cur-style-name">
            <xsl:value-of select="@text:style-name"/>
        </xsl:variable>
        <xsl:variable name="underline" select="//office:automatic-styles/style:style[@style:name=$cur-style-name]/style:text-properties/@style:text-underline-style"/>
        <xsl:variable name="strike" select="//office:automatic-styles/style:style[@style:name=$cur-style-name]/style:text-properties/@style:text-line-through-style"/>
        <xsl:variable name="position" select="//office:automatic-styles/style:style[@style:name=$cur-style-name]/style:text-properties/@style:text-position"/>
        <xsl:variable name="text">
            <xsl:apply-templates/>
        </xsl:variable>
        <xsl:variable name="normalized-text">
            <xsl:value-of select="normalize-space($text)"/>
        </xsl:variable>
        <xsl:if test="$normalized-text != ''">
            <!-- insert space to separate style tags -->
            <xsl:text disable-output-escaping="yes"> </xsl:text>
            <xsl:call-template name="style-font">
                <xsl:with-param name="style">
                    <xsl:value-of select="$cur-style-name"/>
                </xsl:with-param>
            </xsl:call-template>
            <xsl:choose>
                <xsl:when test="$underline='solid'">
                    <xsl:element name="u">
                        <xsl:value-of select="$normalized-text"/>
                    </xsl:element>
                </xsl:when>
                <xsl:when test="$strike='solid'">
                    <xsl:element name="strike">
                        <xsl:value-of select="$normalized-text"/>
                    </xsl:element>
                </xsl:when>
                <xsl:when test="contains($position, 'sub')">
                    <xsl:element name="sub">
                        <xsl:value-of select="$normalized-text"/>
                    </xsl:element>
                </xsl:when>
                <xsl:when test="contains($position, 'super')">
                    <xsl:element name="sup">
                        <xsl:value-of select="$normalized-text"/>
                    </xsl:element>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:value-of select="$normalized-text"/>
                </xsl:otherwise>
            </xsl:choose>
            <xsl:call-template name="style-font">
                <xsl:with-param name="style">
                    <xsl:value-of select="$cur-style-name"/>
                </xsl:with-param>
            </xsl:call-template>
            <!-- insert space to separate style tags -->
            <xsl:text disable-output-escaping="yes"> </xsl:text>
        </xsl:if>
    </xsl:template>

    <!-- line breaks -->
    <xsl:template match="//text:line-break">
        <xsl:choose>
            <xsl:when test="ancestor::table:table-cell">
                <xsl:text disable-output-escaping="yes"> %BR% </xsl:text>
            </xsl:when>
            <xsl:otherwise>
                <xsl:text disable-output-escaping="yes"> %BR%
</xsl:text>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>

    <!-- Group 'Preformatted Text' into a verbatim block -->
    <xsl:template match="text:p[@text:style-name='Preformatted_20_Text']">
        <xsl:choose>
            <!-- don't write verbatim if we are already in the block -->
            <xsl:when test="preceding-sibling::*[1][@text:style-name='Preformatted_20_Text']"/>
            <!-- don't write verbatim if we are inside a table -->
            <xsl:when test="ancestor::table:table-cell">
                <xsl:apply-templates select="." mode="group"/>
            </xsl:when>
            <xsl:otherwise>
                <xsl:element name="verbatim">
                    <xsl:text disable-output-escaping="yes">
</xsl:text>
                    <xsl:apply-templates select="." mode="group"/>
                </xsl:element>
                <xsl:text disable-output-escaping="yes">
</xsl:text>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    <xsl:template match="text:p" mode="group">
        <xsl:choose>
            <!-- we would want a <br> between two text:p in a cell -->
            <xsl:when test="ancestor::table:table-cell">
                <xsl:apply-templates/>
                <!--xsl:element name="br"/-->
                <xsl:text disable-output-escaping="yes"> %BR% </xsl:text>
            </xsl:when>
            <xsl:otherwise>
                <xsl:apply-templates/>
                <xsl:text>
</xsl:text>
            </xsl:otherwise>
        </xsl:choose>
        <xsl:apply-templates select="following-sibling::text:p[1][@text:style-name='Preformatted_20_Text']" mode="group"/>
    </xsl:template>

    <!-- space -->
    <xsl:template match="//text:s">
        <xsl:call-template name="expand-chars">
            <xsl:with-param name="iterations">
                <xsl:value-of select="@text:c"/>
            </xsl:with-param>
            <xsl:with-param name="dup-chars">
                <xsl:text disable-output-escaping="yes"> </xsl:text>
            </xsl:with-param>
        </xsl:call-template>
    </xsl:template>

    <!-- hard tabs -->
    <xsl:template match="//text:tab">
        <xsl:text disable-output-escaping="yes">	</xsl:text>
        <xsl:apply-templates/>
    </xsl:template>

    <!-- Lists - ->
    <xsl:template match="//text:list">
        <xsl:apply-templates/>
    </xsl:template-->
    <xsl:template match="//table:table-cell//text:list">
        <xsl:variable name="cur-style-name">
            <xsl:value-of select="ancestor-or-self::text:list/@text:style-name"/>
        </xsl:variable>
        <xsl:variable name="bullet" select="//office:automatic-styles/text:list-style[@style:name=$cur-style-name]/text:list-level-style-bullet/@text:bullet-char"/>
        <xsl:choose>
            <xsl:when test="$bullet!=''">
                <!-- unescaped space at the end prevents a newline from breaking the table -->
                <xsl:element name="ul"><xsl:text disable-output-escaping="yes"> </xsl:text>
                    <xsl:apply-templates/>
                </xsl:element><xsl:text disable-output-escaping="yes"> </xsl:text>
            </xsl:when>
            <xsl:otherwise>
                <!-- unescaped space at the end prevents a newline from breaking the table -->
                <xsl:element name="ol"><xsl:text disable-output-escaping="yes"> </xsl:text>
                    <xsl:apply-templates/>
                </xsl:element><xsl:text disable-output-escaping="yes"> </xsl:text>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>

    <!-- List items -->
    <xsl:template match="//text:list-item">
        <xsl:variable name="cur-style-name">
            <xsl:value-of select="ancestor::text:list/@text:style-name"/>
        </xsl:variable>
        <xsl:variable name="depth">
            <xsl:value-of select="count(ancestor::text:list)"/>
        </xsl:variable>
        <xsl:variable name="bullet" select="//office:automatic-styles/text:list-style[@style:name=$cur-style-name]/text:list-level-style-bullet[@text:level=$depth]/@text:bullet-char"/>
        <xsl:variable name="number">
            <xsl:value-of select="//office:automatic-styles/text:list-style[@style:name=$cur-style-name]/text:list-level-style-number[@text:level=$depth]/@style:num-format"/>
        </xsl:variable>
        <xsl:variable name="suffix">
            <xsl:value-of select="//office:automatic-styles/text:list-style[@style:name=$cur-style-name]/text:list-level-style-number[@text:level=$depth]/@style:num-suffix"/>
        </xsl:variable>
        <!--xsl:message>current style: <xsl:value-of select="$cur-style-name"/></xsl:message>
        <xsl:message><xsl:value-of select="count(ancestor::text:list)"/> deep</xsl:message>
        <xsl:message>type: <xsl:value-of select="$bullet"/><xsl:value-of select="$number"/></xsl:message-->
        <xsl:choose>
            <!-- can't do proper lists inside table cells -->
            <xsl:when test="ancestor::table:table-cell">
                <!-- unescaped space at the end prevents a newline from breaking the table -->
                <xsl:element name="li"><xsl:text disable-output-escaping="yes"> </xsl:text>
                    <xsl:apply-templates/>
                </xsl:element><xsl:text disable-output-escaping="yes"> </xsl:text>
            </xsl:when>
            <xsl:otherwise>
                <xsl:if test="child::text:p">
                    <xsl:call-template name="expand-chars">
                        <xsl:with-param name="iterations">
                            <xsl:value-of select="count(ancestor::text:list)"/>
                        </xsl:with-param>
                        <xsl:with-param name="dup-chars">
                            <xsl:text disable-output-escaping="yes">	</xsl:text>
                        </xsl:with-param>
                    </xsl:call-template>
                    <xsl:choose>
                        <xsl:when test="$number!=''">
                            <xsl:value-of select='$number'/>
                            <!-- one day TWiki might be able to do this - ->
                            <xsl:value-of select='$suffix'/-->
                            <xsl:if test="$suffix!=''">
                                <xsl:text>.</xsl:text>
                            </xsl:if>
                            <xsl:text disable-output-escaping="yes"> </xsl:text>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:text disable-output-escaping="yes">* </xsl:text>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:if>
                <xsl:apply-templates/>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>

    <!-- Frames -->
    <xsl:template match="//draw:frame">
        <xsl:variable name="cur-style-name">
            <xsl:value-of select="@draw:style-name"/>
        </xsl:variable>
        <xsl:variable name="alignment" select="//office:automatic-styles/style:style[@style:name=$cur-style-name]/style:graphic-properties/@style:horizontal-pos"/>
        <!-- debugging - ->
        <xsl:message>current style: <xsl:value-of select="$cur-style-name"/></xsl:message>
        <xsl:message>alignment: <xsl:value-of select="$alignment"/></xsl:message>
        <xsl:text disable-output-escaping="yes"> </xsl:text>
        <!- - debugging -->
        <xsl:choose>
            <xsl:when test="@text:anchor-type='as-char'">
                <xsl:apply-templates/>
            </xsl:when>
            <xsl:otherwise>
                <xsl:element name="div">
                    <xsl:attribute name="align">
                        <xsl:value-of select="$alignment"/>
                    </xsl:attribute>
                    <xsl:apply-templates/>
                </xsl:element>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>

    <!-- Drawings -->
    <xsl:template match="//draw:g">
        <xsl:text disable-output-escaping="yes">
%X% __Note:__ Could not convert this drawing :-(

</xsl:text>
        <xsl:apply-templates select="descendant::draw:image"/>
    </xsl:template>

    <!-- Images -->
    <!-- This presumes some external script will handle image conversion -->
    <xsl:template match="//draw:image">

        <!-- handle internal/external href, if the hrefs are external they will have the
             prefix prepended to them and the extension will be changed to .png, it's up
             to some kind of wrapper to convert the images there -->
        <xsl:variable name="final-uri">
            <xsl:choose>
                <xsl:when test="starts-with(@xlink:href,'Pictures/')">
                    <xsl:variable name="original-extension">
                        <xsl:value-of select="substring-after(@xlink:href,'.')"/>
                    </xsl:variable>
                    <xsl:choose>
                        <!-- leave gif, jpg or png extensions intact -->
                        <xsl:when test="$original-extension='gif'">
                            <xsl:value-of select="$image-prefix"/><xsl:value-of select="substring(@xlink:href,10)"/>
                        </xsl:when>
                        <xsl:when test="$original-extension='jpg'">
                            <xsl:value-of select="$image-prefix"/><xsl:value-of select="substring(@xlink:href,10)"/>
                        </xsl:when>
                        <xsl:when test="$original-extension='png'">
                            <xsl:value-of select="$image-prefix"/><xsl:value-of select="substring(@xlink:href,10)"/>
                        </xsl:when>
                        <xsl:otherwise>
                            <!-- replace the extension with default so user can
                                convert it and upload it manually -->
                            <xsl:value-of select="$image-prefix"/><xsl:value-of select="substring-before(substring(@xlink:href,10),'.')"/>.<xsl:value-of select="$image-extension"/>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:when>
                <!-- handle object replacement images -->
                <xsl:when test="starts-with(@xlink:href, './ObjectReplacements/')">
                    <xsl:value-of select="$image-prefix"/><xsl:value-of select="translate(substring(@xlink:href,22),' ','_')"/>.<xsl:value-of select="$image-extension"/>
                </xsl:when>
                <xsl:otherwise>
                    <!-- this is an external href, the image is not embedded.
                         We leave it intact in case the user wants to handle this case properly -->
                    <xsl:value-of select="@xlink:href"/>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>

        <xsl:element name="img">
            <xsl:attribute name="src"><xsl:value-of select="$final-uri"/></xsl:attribute>
            <xsl:attribute name="title"><xsl:value-of select="parent::draw:frame/attribute::draw:name"/></xsl:attribute>
            <xsl:attribute name="style">width: <xsl:call-template name="pixels"><xsl:with-param name="value" select="ancestor::draw:frame/attribute::svg:width"/></xsl:call-template>; height: <xsl:call-template name="pixels"><xsl:with-param name="value" select="ancestor::draw:frame/attribute::svg:height"/></xsl:call-template> 
            </xsl:attribute>
        </xsl:element>
        <!--xsl:element name="br"/-->
        <xsl:text disable-output-escaping="yes">
</xsl:text>
        <xsl:apply-templates/>
    </xsl:template>

    <!-- Tables -->
    <xsl:template match="//table:table[not(@table:is-sub-table)]">
    <!-- Uncomment for EDITTABLE style
        <xsl:variable name="table-name" select="@table:name"/>
        <xsl:text disable-output-escaping="yes">

%EDITTABLE{name="</xsl:text>
        <xsl:value-of select="$table-name"/>
        <xsl:text disable-output-escaping="yes">"}%
</xsl:text> -->
        <xsl:apply-templates/>
        <xsl:text disable-output-escaping="yes">
</xsl:text>
    </xsl:template>

    <!-- Table rows -->
    <xsl:template match="//table:table-row">
        <xsl:choose>
            <xsl:when test="descendant::table:table[@table:is-sub-table = 'true']">
                <xsl:variable name="col-span">
                    <!--xsl:value-of select="table:table-cell/@table:number-columns-spanned"/-->
                    <xsl:value-of select="count(table:table-cell/table:table/table:table-column)"/>
                </xsl:variable>
                <xsl:variable name="leading-text">
                    <xsl:text disable-output-escaping="yes">|</xsl:text> 
                    <xsl:choose>
                        <xsl:when test="$col-span > '0'">
                            <xsl:apply-templates select="table:table-cell[not(child::table:table)] | table:covered-table-cell[count(preceding-sibling::table:covered-table-cell) >= ($col-span - 1)]"/>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:apply-templates select="table:table-cell[not(child::table:table)] | table:covered-table-cell"/>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:variable>
                <xsl:variable name="dup-col-span">
                    <xsl:choose>
                        <xsl:when test="$col-span > '0'">
                            <xsl:value-of select="count(table:table-cell[not(child::table:table)] | table:covered-table-cell[count(preceding-sibling::table:covered-table-cell) >= ($col-span - 1)])"/>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:value-of select="count(table:table-cell[not(child::table:table)] | table:covered-table-cell)"/>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:variable>
                <xsl:variable name="dup-text">
                    <xsl:text>|</xsl:text>
                    <xsl:call-template name="expand-chars">
                        <xsl:with-param name="iterations" select="$dup-col-span"/>
                        <xsl:with-param name="dup-chars">
                            <xsl:text disable-output-escaping="yes">^|</xsl:text>
                        </xsl:with-param>
                    </xsl:call-template>
                </xsl:variable>
                <!--xsl:message>dup-col-span: <xsl:value-of select="$dup-col-span"/></xsl:message>
                <xsl:message>dup-text: <xsl:value-of select="$dup-text"/></xsl:message-->
                <xsl:for-each select="descendant::table:table[@table:is-sub-table]/table:table-row">
                    <xsl:choose>
                        <xsl:when test="not(preceding-sibling::table:table-row)">
                            <xsl:value-of select="$leading-text"/>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:value-of select="$dup-text"/>
                        </xsl:otherwise>
                    </xsl:choose>
                    <xsl:apply-templates select="."/>
                </xsl:for-each>
            </xsl:when>
            <xsl:otherwise>
                <xsl:text disable-output-escaping="yes">|</xsl:text> 
                <xsl:apply-templates/>
                <xsl:text disable-output-escaping="yes">
</xsl:text>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    <xsl:template match="//table:table[@table:is-sub-table]/table:table-row">
        <!--xsl:text disable-output-escaping="yes">sub|</xsl:text-->
        <xsl:apply-templates/>
        <xsl:text disable-output-escaping="yes">
</xsl:text>
    </xsl:template>

    <!-- Table cells -->
    <xsl:template match="//table:table-cell">
        <xsl:variable name="cur-style-name">
            <xsl:value-of select="child::text:p/@text:style-name"/>
        </xsl:variable>
        <xsl:variable name="alignment" select="//office:automatic-styles/style:style[@style:name=$cur-style-name]/style:paragraph-properties/@fo:text-align"/>
        <!-- debug - ->
        <xsl:message>current style: <xsl:value-of select="$cur-style-name"/></xsl:message>
        <xsl:message>alignment: <xsl:value-of select="$alignment"/></xsl:message>
        <xsl:text disable-output-escaping="yes"> </xsl:text>
        <!- - debug -->
        <xsl:text disable-output-escaping="yes"> </xsl:text>
        <xsl:choose>
            <xsl:when test="$alignment='center'">
        		<xsl:text disable-output-escaping="yes">  </xsl:text>
		        <xsl:apply-templates/>
        		<xsl:text disable-output-escaping="yes">  </xsl:text>
            </xsl:when>
            <xsl:when test="$alignment='end'">
        		<xsl:text disable-output-escaping="yes">  </xsl:text>
		        <xsl:apply-templates/>
            </xsl:when>
            <xsl:otherwise>
		        <xsl:apply-templates/>
            </xsl:otherwise>
        </xsl:choose>
        <xsl:text disable-output-escaping="yes"> |</xsl:text>
    </xsl:template>
    <!-- Handles horizontally merged cells -->
    <xsl:template match="//table:covered-table-cell">
        <xsl:text disable-output-escaping="yes">|</xsl:text>
    </xsl:template>

    <!-- Table of Contents -->
    <xsl:template match="//text:table-of-content">
        <xsl:text disable-output-escaping="yes">
%TOC%

</xsl:text>
    </xsl:template>

    <!-- Headings -->
    <xsl:template match="//text:h">
        <xsl:variable name="plusses">
            <xsl:call-template name="expand-chars">
                <xsl:with-param name="iterations">
                    <xsl:value-of select="@text:outline-level"/>
                </xsl:with-param>
                <xsl:with-param name="dup-chars">
                    <xsl:text disable-output-escaping="yes">+</xsl:text>
                </xsl:with-param>
            </xsl:call-template>
        </xsl:variable>
        <xsl:variable name="bookmarks">
            <xsl:for-each select="text:reference-mark | text:bookmark | text:bookmark-start">
                <xsl:element name="a">
                    <xsl:attribute name="name">
                        <xsl:value-of select="@text:name"/>
                    </xsl:attribute>
                    <xsl:text> </xsl:text>
                </xsl:element>
            </xsl:for-each>
        </xsl:variable>
        <xsl:text disable-output-escaping="yes">
</xsl:text>
        <xsl:copy-of select="$bookmarks"/>
        <xsl:text disable-output-escaping="yes">
---</xsl:text><xsl:value-of select="$plusses"/><xsl:text disable-output-escaping="yes"> </xsl:text>
        <xsl:apply-templates/>
        <xsl:text disable-output-escaping="yes">
</xsl:text>
    </xsl:template>

    <!-- NOTOC headings -->
    <!-- Is this even possible? I seem to remember seeing it, but couldn't reproduce it -->
    <xsl:template match="//text:p[@text:style-name='Heading_20_1']">
        <xsl:text disable-output-escaping="yes">
---+!! </xsl:text>
        <xsl:apply-templates/>
        <xsl:text disable-output-escaping="yes">
</xsl:text>
    </xsl:template>
    <xsl:template match="//text:p[@text:style-name='Heading_20_2']">
        <xsl:text disable-output-escaping="yes">
---++!! </xsl:text>
        <xsl:apply-templates/>
        <xsl:text disable-output-escaping="yes">
</xsl:text>
    </xsl:template>
    <xsl:template match="//text:p[@text:style-name='Heading_20_3']">
        <xsl:text disable-output-escaping="yes">
---+++!! </xsl:text>
        <xsl:apply-templates/>
        <xsl:text disable-output-escaping="yes">
</xsl:text>
    </xsl:template>
    <xsl:template match="//text:p[@text:style-name='Heading_20_4']">
        <xsl:text disable-output-escaping="yes">
---++++!! </xsl:text>
        <xsl:apply-templates/>
        <xsl:text disable-output-escaping="yes">
</xsl:text>
    </xsl:template>
    <xsl:template match="//text:p[@text:style-name='Heading_20_5']">
        <xsl:text disable-output-escaping="yes">
---+++++!! </xsl:text>
        <xsl:apply-templates/>
        <xsl:text disable-output-escaping="yes">
</xsl:text>
    </xsl:template>
    <xsl:template match="//text:p[@text:style-name='Heading_20_6']">
        <xsl:text disable-output-escaping="yes">
---++++++!! </xsl:text>
        <xsl:apply-templates/>
        <xsl:text disable-output-escaping="yes">
</xsl:text>
    </xsl:template>
    <xsl:template match="//text:p[@text:style-name='Heading_20_7']">
        <xsl:text disable-output-escaping="yes">
---+++++++!! </xsl:text>
        <xsl:apply-templates/>
        <xsl:text disable-output-escaping="yes">
</xsl:text>
    </xsl:template>
    <xsl:template match="//text:p[@text:style-name='Heading_20_8']">
        <xsl:text disable-output-escaping="yes">
---++++++++!! </xsl:text>
        <xsl:apply-templates/>
        <xsl:text disable-output-escaping="yes">
</xsl:text>
    </xsl:template>
    <xsl:template match="//text:p[@text:style-name='Heading_20_9']">
        <xsl:text disable-output-escaping="yes">
---+++++++++!! </xsl:text>
        <xsl:apply-templates/>
        <xsl:text disable-output-escaping="yes">
</xsl:text>
    </xsl:template>
    <xsl:template match="//text:p[@text:style-name='Heading_20_10']">
        <xsl:text disable-output-escaping="yes">
---++++++++++!! </xsl:text>
        <xsl:apply-templates/>
        <xsl:text disable-output-escaping="yes">
</xsl:text>
    </xsl:template>

    <!-- Bookmarks -->
    <xsl:template match="//text:a">
        <xsl:text>[[</xsl:text>
        <xsl:value-of select="@xlink:href"/>
        <xsl:text>][</xsl:text>
        <xsl:value-of select="."/>
        <xsl:text>]]</xsl:text>
    </xsl:template>
    <xsl:template match="//text:bookmark-ref | //text:reference-ref">
        <xsl:text>[[#</xsl:text>
        <xsl:value-of select="@text:ref-name"/>
        <xsl:text>][</xsl:text>
        <xsl:value-of select="."/>
        <xsl:text>]]</xsl:text>
    </xsl:template>
    <xsl:template match="//text:bookmark | //text:bookmark-start | //text:reference-mark">
        <xsl:element name="a">
            <xsl:attribute name="name">
                <xsl:value-of select="@text:name"/>
            </xsl:attribute>
            <xsl:text> </xsl:text>
        </xsl:element>
    </xsl:template>

    <!-- handle bookmarks in headings template or it breaks the TOC -->
    <xsl:template match="//text:h/text:bookmark"/>
    <xsl:template match="//text:h/text:bookmark-start"/>
    <xsl:template match="//text:h/text:bookmark-end"/>
    <xsl:template match="//text:h/text:reference-mark"/>

    <!-- Tracked changes -->
    <xsl:template match="text:tracked-changes"/>

    <!-- call by name stuff -->
    <xsl:template name="insert-style">
        <xsl:call-template name="insert-style-work">
            <xsl:with-param name="stylename" select="@text:style-name"/>
        </xsl:call-template>
   
        <xsl:call-template name="insert-style-work">
            <xsl:with-param name="stylename" select="@draw:style-name"/>
        </xsl:call-template>
   
        <xsl:call-template name="insert-style-work">
            <xsl:with-param name="stylename" select="@table:style-name"/>
        </xsl:call-template>
    </xsl:template>

    <xsl:template name="insert-style-work">
        <xsl:param name="stylename"/>
        <xsl:param name="final-stylename" select="translate($stylename,'. ','_')"/>
   
        <xsl:if test="$stylename != ''">
            <xsl:attribute name="class">
                <xsl:value-of select="$style-prefix"/>_<xsl:value-of select="$final-stylename"/>
            </xsl:attribute>
        </xsl:if>
    </xsl:template>

    <xsl:template name="generate-style">
        <xsl:param name="stylename"/>
        <!-- cleanse name -->
        <xsl:param name="final-stylename" select="translate($stylename,'. ','_')"/>

        <xsl:if test="$stylename">
            .<xsl:value-of select="$style-prefix"/> .<xsl:value-of select="$style-prefix"/>_<xsl:value-of select="$final-stylename"/> {
            <xsl:apply-templates mode="stylesheet"/>}
        </xsl:if>
    </xsl:template>

    <!-- expand spaces for preformatted text -->
    <!-- expand tabs for lists -->
    <!-- expand plusses for headings -->
    <!-- expand carats for vertically merged table cells -->
    <xsl:template name="expand-chars">
        <xsl:param name="iterations"/>
        <xsl:param name="i">1</xsl:param>
        <xsl:param name="string"/>
        <xsl:param name="dup-chars"/>

        <xsl:choose>
            <!-- loop -->
            <xsl:when test="$i &lt; $iterations">
                <xsl:call-template name="expand-chars">
                    <xsl:with-param name="type"/>
                    <xsl:with-param name="iterations" select="$iterations"/>
                    <xsl:with-param name="dup-chars" select="$dup-chars"/>
                    <xsl:with-param name="i" select="$i + 1"/>
                    <xsl:with-param name="string" select="concat($string,$dup-chars)"/>
                </xsl:call-template>
            </xsl:when>

            <!-- done -->
            <xsl:otherwise>
                <xsl:value-of select="concat($string,$dup-chars)"/>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>

    <!-- wow this sucks -->
    <xsl:template name="nbsp">
        <xsl:text disable-output-escaping="yes">&amp;nbsp;</xsl:text>
    </xsl:template>

    <!-- modeled from sx2ml -->
    <xsl:template name="pixels">
        <xsl:param name="value"/>

        <xsl:variable name="cm-in-mm" select="10" />
        <xsl:variable name="in-in-mm" select="25.4" />
        <xsl:variable name="px-in-mm" select="$in-in-mm div $dpi" />

        <xsl:variable name="no-of-mm" select="number(translate($value,'cminh', '')) div $px-in-mm"/>

        <xsl:choose>
            <xsl:when test="contains($value, 'mm')">
                <xsl:value-of select="ceiling($no-of-mm * 1)"/>px</xsl:when>
            <xsl:when test="contains($value, 'cm')">
                <xsl:value-of select="ceiling($no-of-mm * $cm-in-mm)"/>px</xsl:when>
            <xsl:when test="contains($value, 'in')">
                <xsl:value-of select="ceiling($no-of-mm * $in-in-mm)"/>px</xsl:when>
            <xsl:otherwise>
                <!-- not in/cm/mm, just return value -->
                <xsl:value-of select="$value" /></xsl:otherwise>
        </xsl:choose>
    </xsl:template>

</xsl:stylesheet>
<!-- vim:ts=4:sw=4:et:tw=0
-->

