# Module of TWiki Enterprise Collaboration Platform, http://TWiki.org/
#
# Copyright (C) 2006-2007 Sven Dowideit, SvenDowideit@distributedINFORMATION.com
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version. For
# more details read LICENSE in the root of this distribution.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
# As per the GPL, removal of this notice is prohibited.

=begin twiki

---+ package TWiki::Users::OpenIDUserMapping

uses Net::OpenID11 to implement an OpenID consumer that uses the 
'Simple registration Extensions 1.0'

=cut

package TWiki::Users::OpenIDUserMapping;

use strict;
use strict;
use Assert;
use TWiki::Users::TWikiUserMapping;
use TWiki::Time;
use Net::OpenID11::Consumer;
use Net::OpenID11::Stores::FileStore;
use TWiki::ListIterator;

use Error qw( :try );

@TWiki::Users::OpenIDUserMapping::ISA = qw( TWiki::Users::TWikiUserMapping );

=pod

---++ ClassMethod new( $session ) -> $object

Constructs a new user mapping handler of this type, referring to $session
for any required TWiki services.

=cut

sub new {
    my ( $class, $session ) = @_;

    my $this = bless( $class->SUPER::new($session, 'OpenIDUserMapping_'), $class );
    $this->{session}    = $session;
    $this->{mapping_id} = 'OpenIDUserMapping_';

    return $this;
}

=begin twiki

---++ ObjectMethod finish()
Break circular references.

=cut

# Note to developers; please undef *all* fields in the object explicitly,
# whether they are references or not. That way this method is "golden
# documentation" of the live fields in the object.
sub finish {
    my $this = shift;
    $this->_saveUsers();

    undef $this->{U2W};
    undef $this->{W2U};
    undef $this->{L2U};
    undef $this->{U2L};
    undef $this->{U2E};

    $this->SUPER::finish();
}

=pod

---++ ObjectMethod loginTemplateName () -> templateFile

allows UserMappings to come with customised login screens - that should preffereably only over-ride the UI function

=cut

sub loginTemplateName {
    return 'login.openid';
}

=pod

---++ ObjectMethod supportsRegistration () -> false
return 1 if the UserMapper supports registration (ie can create new users)

=cut

sub supportsRegistration {
    return;
}

=pod

---++ ObjectMethod handlesUser ( $cUID, $login, $wikiname) 

Called by the TWiki::User object to determine which loaded mapping to use for a given user (must be fast)
in the BaseUserMapping case, we know all the users we deal specialise in.

=cut

sub handlesUser {
    my ( $this, $cUID, $login, $wikiname ) = @_;
    
    $cUID = '' unless (defined($cUID));
    $login = '' unless (defined($login));
    $wikiname = '' unless (defined($wikiname));
    
    #if its a URL, its for us.
    return 1 if ($login =~ $TWiki::cfg{LinkProtocolPattern});
    return 1 if ($cUID =~ /^($this->{mapping_id})/ );

    #the remainder of the tests check if we have known this user before    
    $this->_loadUsers();
    return 1 if ($wikiname && $this->findUserByWikiName( $wikiname ));
    #return 1 if ( $login && $this->getLoginName($login) );

    return 0;
}

=pod

---++ ObjectMethod getCanonicalUserID ($login) -> cUID

Convert a login name to the corresponding canonical user name. The
canonical name can be any string of 7-bit alphanumeric and underscore
characters, and must correspond 1:1 to the login name.
(undef on failure)

=cut

sub getCanonicalUserID {
    my ( $this, $login, $dontcheck ) = @_;
    return unless ( defined($login) && ( $login ne '' ) );
    return $login if ( $login =~ /^($this->{mapping_id})/ );
    
    $this->_loadUsers();    

    unless ( defined( $this->{gotSession} ) ) {
        my $cUID =
          $this->{session}->{users}->{loginManager}->getSessionValue('cUID');

        unless ( defined($cUID) && ( $cUID ne '' ) ){
            $cUID = $login;
            use bytes;
    
            # use bytes to ignore character encoding
            $cUID =~ s/([^a-zA-Z0-9_])/'_'.sprintf('%02d', ord($1))/ge;
            no bytes;
            $cUID = $this->{mapping_id} . $cUID;
        }

        my $authUser =
          $this->{session}->{users}->{loginManager}
          ->getSessionValue('AUTHUSER');
        my $wikiName =
          $this->{session}->{users}->{loginManager}
          ->getSessionValue('WIKINAME');

        my $email =
          $this->{session}->{users}->{loginManager}->getSessionValue('EMAIL');

        $this->_addUser($cUID, $authUser, $wikiName, $email) if (defined($authUser));

        $this->{gotSession} = 1;
    }

    return $this->{L2U}->{$login} if ( defined( $this->{L2U}->{$login} ) );
    return unless ( ($dontcheck) || defined( $this->{L2U}->{$login} ) );

    use bytes;

    # use bytes to ignore character encoding
    $login =~ s/([^a-zA-Z0-9_])/'_'.sprintf('%02d', ord($1))/ge;
    no bytes;
    $login = $this->{mapping_id} . $login;
    return $login;
}

=pod

---++ ObjectMethod getLoginName ($cUID) -> login

converts an internal cUID to that user's login
(undef on failure)

=cut

sub getLoginName {
    my ( $this, $user ) = @_;
    ASSERT($user)                 if DEBUG;
    ASSERT( $this->{mapping_id} ) if DEBUG;

    $this->_loadUsers();
    return $this->{U2L}->{$user};
}

=pod

---++ ClassMethod addUser ($login, $wikiname) -> cUID

no registration, this is a read only user mapping
throws an Error::Simple 

Add a user to the persistant mapping that maps from usernames to wikinames
and vice-versa. The default implementation uses a special topic called
"TWikiUsers" in the users web. Subclasses will provide other implementations
(usually stubs if they have other ways of mapping usernames to wikinames).
Names must be acceptable to $TWiki::cfg{NameFilter}
$login must *always* be specified. $wikiname may be undef, in which case
the user mapper should make one up.
This function must return a *canonical user id* that it uses to uniquely
identify the user. This can be the login name, or the wikiname if they
are all guaranteed unigue, or some other string consisting only of 7-bit
alphanumerics and underscores.
if you fail to create a new user (for eg your Mapper has read only access), 
            throw Error::Simple(
               'Failed to add user: '.$ph->error());

=cut

sub addUser {
    my ( $this, $login, $wikiname ) = @_;

    ASSERT($login) if DEBUG;

    throw Error::Simple('user creation is not supported by the BaseUserMapper');
    return 0;
}

=pod

---++ ObjectMethod removeUser( $user ) -> $boolean

no registration, this is a read only user mapping
throws an Error::Simple 

=cut

sub removeUser {
    throw Error::Simple('user removal is not supported by the BaseUserMapper');
    return 0;
}

=pod

---++ ObjectMethod getWikiName ($cUID) -> wikiname

# Map a canonical user name to a wikiname

=cut

sub getWikiName {
    my ( $this, $cUID ) = @_;
    
    $this->_loadUsers();
    return $this->{U2W}->{$cUID} || getLoginName( $this, $cUID );
}

=pod

---++ ObjectMethod userExists( $user ) -> $boolean

Determine if the user already exists or not.

=cut

sub userExists {
    my ( $this, $cUID ) = @_;
    $this->ASSERT_IS_CANONICAL_USER_ID($cUID) if DEBUG;

    $this->_loadUsers();
    return $this->{U2L}->{$cUID};
}

=pod

---++ ObjectMethod eachUser () -> listIterator of cUIDs

Called from TWiki::Users. See the documentation of the corresponding
method in that module for details.

=cut

sub eachUser {
    my ($this) = @_;

    $this->_loadUsers();
    my @list = keys( %{ $this->{U2W} } );
    return new TWiki::ListIterator( \@list );
}

=pod

---++ ObjectMethod eachGroupMember ($group) ->  listIterator of cUIDs

Called from TWiki::Users. See the documentation of the corresponding
method in that module for details.

=cut

sub eachGroupMember {
    my $this  = shift;
    my $group = shift;

    #TODO: implemend expanding of nested groups
    my $members = $this->{GROUPS}{$group};

    #print STDERR "eachGroupMember($group): ".join(',', @{$members});

    return new TWiki::ListIterator($members);
}

=pod

---++ ObjectMethod isGroup ($user) -> boolean
TODO: what is $user - wikiname, UID ??
Called from TWiki::Users. See the documentation of the corresponding
method in that module for details.

=cut

sub isGroup {
    my ( $this, $user ) = @_;

    #TODO: what happens to the code if we implement this using an iterator too?
    return grep( /$user/, $this->eachGroup() );
}

=pod

---++ ObjectMethod eachGroup () -> ListIterator of groupnames

Called from TWiki::Users. See the documentation of the corresponding
method in that module for details.

=cut

sub eachGroup {
    my ($this) = @_;
    my @groups = keys( %{ $this->{GROUPS} } );

    return new TWiki::ListIterator( \@groups );
}

=pod

---++ ObjectMethod eachMembership ($cUID) -> ListIterator of groups this user is in

Called from TWiki::Users. See the documentation of the corresponding
method in that module for details.

=cut

sub eachMembership {
    my ( $this, $cUID ) = @_;

    my $it = $this->eachGroup();
    $it->{filter} = sub {
        $this->isInGroup( $cUID, $_[0] );
    };
    return $it;
}

=pod

---++ ObjectMethod isAdmin( $cUID ) -> $boolean

True if the user is an admin
   * is a member of the $TWiki::cfg{SuperAdminGroup}

=cut

sub isAdmin {
    my ( $this, $cUID ) = @_;

    return
      0;  #openID does not implement groups - it would have to happen via topics

    my $isAdmin = 0;
    $this->ASSERT_IS_CANONICAL_USER_ID($cUID) if DEBUG;

    my $sag = $TWiki::cfg{SuperAdminGroup};
    $isAdmin = $this->isInGroup( $cUID, $sag );

    return $isAdmin;
}

=pod

---++ ObjectMethod isInGroup ($user, $group, $scanning) -> bool

Called from TWiki::Users. See the documentation of the corresponding
method in that module for details.

=cut

sub isInGroup {
    my ( $this, $user, $group, $scanning ) = @_;
    ASSERT($user) if DEBUG;

    my @users;
    my $it = $this->eachGroupMember($group);
    while ( $it->hasNext() ) {
        my $u = $it->next();
        next if $scanning->{$u};
        $scanning->{$u} = 1;
        return 1 if $u eq $user;
        if ( $this->isGroup($u) ) {
            return 1 if $this->isInGroup( $user, $u, $scanning );
        }
    }
    return 0;
}

=pod

---++ ObjectMethod findUserByEmail( $email ) -> \@users
   * =$email= - email address to look up
Return a list of canonical user names for the users that have this email
registered with the password manager or the user mapping manager.

The password manager is asked first for whether it maps emails.
If it doesn't, then the user mapping manager is asked instead.

=cut

sub findUserByEmail {
    my ( $this, $email ) = @_;

    throw Error::Simple('IMPLEMENT ME TWiki::BaseUserMapping');
}

=pod

---++ ObjectMethod getEmails($user) -> @emailAddress

If this is a user, return their email addresses. If it is a group,
return the addresses of everyone in the group.

The password manager and user mapping manager are both consulted for emails
for each user (where they are actually found is implementation defined).

Duplicates are removed from the list.

=cut

sub getEmails {
    my ( $this, $user ) = @_;
    $this->ASSERT_IS_CANONICAL_USER_ID($user) if DEBUG;

    return $this->{U2E}->{$user} || ();
}

=pod

---++ ObjectMethod setEmails($user, @emails)

Set the email address(es) for the given user.
The password manager is tried first, and if it doesn't want to know the
user mapping manager is tried.

=cut

sub setEmails {
    my $this = shift;
    my $user = shift;

    throw Error::Simple(
        'setting emails is not supported by the BaseUserMapper');
    return 0;
}

=pod

---++ ObjectMethod findUserByWikiName ($wikiname) -> list of cUIDs associated with that wikiname

Called from TWiki::Users. See the documentation of the corresponding
method in that module for details.

=cut

sub findUserByWikiName {
    my ( $this, $wn ) = @_;
    my @users = ();

    if ( $this->isGroup($wn) ) {
        push( @users, $wn );
    }
    else {

        # Add additional mappings defined in TWikiUsers
        if ( $this->{W2U}->{$wn} ) {
            push( @users, $this->{W2U}->{$wn} );
        }
        else {

            # Bloody compatibility!
            # The wikiname is always a registered user for the purposes of this
            # mapping. We have to do this because TWiki defines access controls
            # in terms of mapped users, and if a wikiname is *missing* from the
            # mapping there is "no such user".
            push( @users, getCanonicalUserID( $this, $wn ) );
        }

        #    } else {
        # The wikiname is also the login name, so we can just convert
        # it to a canonical user id
        #        push( @users, getCanonicalUserID( $this, $wn ));
    }
    return \@users;
}

=pod

---++ ObjectMethod checkPassword( $userName, $passwordU ) -> $boolean

Finds if the password is valid for the given user.

Returns 1 on success, undef on failure.

=cut

sub checkPassword {
    my ( $this, $user, $password ) = @_;

    my $twiki = $this->{session};
    my $query = $twiki->{cgiQuery};
#TODO: move this code to OpenIDLogin::login - or at least docco why i didn't (I htink it was due to a 4.1.2 issue)
    $user     = TWiki::Sandbox::untaintUnchecked($user);
    $password = TWiki::Sandbox::untaintUnchecked($password);

    my $STORE_DIR = $TWiki::cfg{WorkingDir}. '/openid/store';
    my $SESSION_DIR = $TWiki::cfg{WorkingDir}. '/openid/session';

    my $session =
      new CGI::Session( undef, $query, { Directory => $SESSION_DIR } );
    my $cookie = $session->cookie( CGISESSID => $session->id );

    my $store = Net::OpenID11::Stores::FileStore->new($STORE_DIR);
    my $consumer = Net::OpenID11::Consumer->new( $session, $store );

    my $user_url = $user;    #'http://svendowideit.home.org.au';
    if ($user_url) {         # Begin OpenID transaction
        my $request = $consumer->begin($user_url);
        if ( $request->status eq 'failure' )
        {                    # this is an unrecoverable discovery failure
            print STDERR "OPENID AUTH ERROR: ".
                $request->{message}
              . ' authenticating '
              . $request->{identity_url};    #TODO: make into an oops
                                             #display_failure($request);
            return;
        }
        else {                               # Redirect to OpenID server
            $request->addExtensionArg( 'sreg', "required", "fullname,email" );
            my $trust_root = $this->{session}->{urlHost};
            my $origurl    = $query->param('origurl');
            my $topic      = $twiki->{topicName};
            my $web        = $twiki->{webName};
            #TODO: return to should include the origurl, or the TWIKISSID so an edit login goes to edit.
            my $return_to  = $this->{session}->getScriptUrl( 1, 'login', $web, $topic );
            $origurl = '<empty>' unless ($origurl);
            #print STDERR
            #    "trust_root: $trust_root ; origurl: $origurl ; return_to: $return_to ;";
            my $redirect_url = $request->redirectURL( $trust_root, $return_to );
            print $query->header( -cookie => $cookie,
                -location => $redirect_url );
            return;
        }
    }

    die "we should never get here";
    return;
}

=pod

---++ ObjectMethod setPassword( $user, $newPassU, $oldPassU ) -> $boolean

If the $oldPassU matches matches the user's password, then it will
replace it with $newPassU.

If $oldPassU is not correct and not 1, will return 0.

If $oldPassU is 1, will force the change irrespective of
the existing password, adding the user if necessary.

Otherwise returns 1 on success, undef on failure.

=cut

sub setPassword {
    my ( $this, $user, $newPassU, $oldPassU ) = @_;
    $this->ASSERT_IS_CANONICAL_USER_ID($user) if DEBUG;
    throw Error::Simple(
        'cannot change user passwords using TWiki::BaseUserMapping');
}

=pod

---++ ObjectMethod passwordError( ) -> $string

returns a string indicating the error that happened in the password handlers
TODO: these delayed error's should be replaced with Exceptions.

returns undef if no error

=cut

sub passwordError {
    my $this = shift;

    return $this->{error};
}

=pod

---++ ObjectMethod ASSERT_IS_CANONICAL_USER_ID( $user_id ) -> $boolean

used for debugging to ensure we are actually passing a canonical_id

=cut

sub ASSERT_IS_CANONICAL_USER_ID {
    my ( $this, $user_id ) = @_;

    #print STDERR "ASSERT_IS_CANONICAL_USER_ID($user_id)";
    #    ASSERT($user_id =~/^UID$(\s+)UID$/) if DEBUG;
    ASSERT( $user_id =~ /^$this->{mapping_id}/ )
      ;    #refine with more specific regex

}

=pod

---++ ObjectMethod ASSERT_IS_USER_LOGIN_ID( $user_login ) -> $boolean

used for debugging to ensure we are actually passing a user login

=cut

sub ASSERT_IS_USER_LOGIN_ID {
    my ( $this, $user_login ) = @_;
    1;
}

=pod

---++ ObjectMethod ASSERT_IS_USER_DISPLAY_NAME( $user_display ) -> $boolean

used for debugging to ensure we are actually passing a user display_name (commonly a WikiWord Name)

=cut

sub ASSERT_IS_USER_DISPLAY_NAME {
    my ( $this, $user_display ) = @_;
    1;
}

sub _loadUsers {
    my $this = shift;

    ASSERT(UNIVERSAL::isa($this,"TWiki::Users::OpenIDUserMapping")) if DEBUG;
    ASSERT($this->can('addUser')) if DEBUG;
   
    unless (defined($this->{userStoreLoadedFromFile})) {
#print STDERR "_loadUsers\n";
        my $workarea = $TWiki::cfg{DataDir}.'/OpenIDUsers.txt';
        my $users = TWiki::Func::readFile($workarea);
        
        my @lines = split(/\n/, $users);
        foreach my $line (@lines) {
            my ($cUID, $login, $wikiName, $emails) = split(/\s*[|]\s*/, $line);
            $this->_addUser($cUID, $login, $wikiName, $emails);
        }
        $this->{userStoreLoadedFromFile} = 1;
    }
}

sub _saveUsers {
    my $this = shift;
    return unless ((defined($this->{userStoreLoadedFromFile})) && ($this->{userStoreLoadedFromFile} > 1));
    
#print STDERR "_saveUsers\n";
    my $workarea = $TWiki::cfg{DataDir}.'/OpenIDUsers.txt';
    my $users = join("\n", map(
                               {
                                    my $login = $_;
                                    my $cUID = $this->{L2U}->{$login} || '';
                                    my $wikiName = $this->{U2W}->{$cUID} || '';
                                    my $emails = $this->{U2E}->{$cUID} || '';
                                    
                                    "$cUID | $login | $wikiName | $emails";
                                }
                                keys( %{ $this->{L2U} } )
                              )
                    );
    
    TWiki::Func::saveFile($workarea, $users);
}

sub _addUser {
    my ($this, $cUID, $login, $wikiName, $emails) = @_;
    
    ASSERT(UNIVERSAL::isa($this,"TWiki::Users::OpenIDUserMapping")) if DEBUG;
    ASSERT($this->can('addUser')) if DEBUG;
    
    return unless ( $cUID =~ /^($this->{mapping_id})/ );
    return unless (defined($login));
    return unless (defined($wikiName));
    
    $emails = '' unless (defined($emails));
    
    #print STDERR "_addUser(".($cUID||'undef').", ".($login||'undef').", ".($wikiName||'undef').", ".($emails||'undef').")\n";
    if (
        #was not yet defined
        (!defined($this->{U2W}->{$cUID}))
        ||
        #values have changed
        (! ((defined($this->{U2W}->{$cUID})) &&
        ($this->{U2W}->{$cUID} eq $wikiName) &&
        ($this->{L2U}->{$login} eq $cUID) &&
        ($this->{U2E}->{$cUID} eq $emails)) )
       ) {
        $this->{U2W}->{$cUID} = $wikiName;
        $this->{W2U}->{$wikiName} = $cUID;
        $this->{L2U}->{$login} = $cUID;
        $this->{U2L}->{$cUID} = $login;
        $this->{U2E}->{$cUID} = $emails;
        if (defined($this->{userStoreLoadedFromFile})) {
            $this->{userStoreLoadedFromFile}++;         #signal to '_saveUsers'
        }
    }
}

1;
