# Module of TWiki Enterprise Collaboration Platform, http://TWiki.org/
#
# Copyright (C) 2006 Sven Dowideit, SvenDowideit@home.org.au
# and TWiki Contributors. All Rights Reserved. TWiki Contributors
# are listed in the AUTHORS file in the root of this distribution.
# NOTE: Please extend that file, not this notice.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version. For
# more details read LICENSE in the root of this distribution.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
# As per the GPL, removal of this notice is prohibited.

=pod

---+ package TWiki::LoginManager::OpenIDLogin

This is a login manager that you can specify in the security setup section of [[%SCRIPTURL%/configure%SCRIPTSUFFIX%][configure]].
 It provides users with a template-based form to enter usernames and passwords, and works with the PasswordManager that you specify to verify those passwords.

Subclass of TWiki::LoginManager; see that class for documentation of the
methods of this class.

=cut

package TWiki::LoginManager::OpenIDLogin;

use strict;
use Assert;
use TWiki::LoginManager::TemplateLogin;
use File::Spec;
use Net::OpenID11::Consumer;
use Net::OpenID11::Stores::FileStore;

@TWiki::LoginManager::OpenIDLogin::ISA = ('TWiki::LoginManager::TemplateLogin');

sub new {
    my ( $class, $session ) = @_;

    #no spaces restriction makes no sense in this context
    #$TWiki::cfg{LoginNameFilterIn} = qr/^[^\*?~^\$@%`"'&;|<>\x00-\x1f]+$/;    #`
    $TWiki::cfg{LoginNameFilterIn} = $TWiki::cfg{LinkProtocolPattern};#qr/^[^\*?~^\$@%`"'&;|<>\x00-\x1f]+$/;

    my $this = bless( $class->SUPER::new($session), $class );
    $session->enterContext('can_login');
    
    # over-ride registered tag handlers and values
    TWiki::registerTagHandler('LOGIN', \&_LOGIN);
    
    return $this;
}

#add a form input field so users can log in directly, bypassing the 'login' screen
sub _LOGIN {
    #my( $twiki, $params, $topic, $web ) = @_;
    my $twiki = shift;
    my $this = $twiki->{users}->{loginManager};

    return '' if $twiki->inContext( 'authenticated' );

    my $url = $this->loginUrl();
    if( $url ) {
        #load the login tmpl
        $twiki->templates->readTemplate('login');
        $twiki->templates->readTemplate($twiki->{users}->loginTemplateName());
        
        $twiki->addToHEAD('openidcss', $twiki->templates->expandTemplate('openidcss').
                          $twiki->templates->expandTemplate('leftbarlogincss'));
        
        my $text = $twiki->templates->expandTemplate('leftbarlogin');
        return "$text";
    }
    return '';
}

sub login {
    my( $this, $query, $twikiSession ) = @_;
    my $twiki = $this->{twiki};
    my $users = $twiki->{users};

    my $origurl = $query->param( 'origurl' );
    my $loginName = $query->param( 'username' );
    my $loginPass = $query->param( 'password' );
    my $remember = $query->param( 'remember' );
    
    my $wikiName  = '';
    my $cUID      = '';
    my $authLogin = '';
    my $authUser  = '';
    my $email     = '';

    #is it an openID Auth?
    return $this->SUPER::login($query, $twikiSession)
        unless ( defined( $twiki->{cgiQuery}->param('openid.mode') ));

    my $STORE_DIR = $TWiki::cfg{WorkingDir}. '/openid/store';
    my $SESSION_DIR = $TWiki::cfg{WorkingDir}. '/openid/session';

    my $session =
      new CGI::Session( undef, $query, { Directory => $SESSION_DIR } );
    my $cookie = $query->cookie( CGISESSID => $session->id );

    my $store = Net::OpenID11::Stores::FileStore->new($STORE_DIR);
    my $consumer = Net::OpenID11::Consumer->new( $session, $store );

    #complete the openid auth request started in OpenIDUserMapping::checkPassword
    my %query    = $query->Vars;
    my $response = $consumer->complete( \%query );
    if ( $response->status eq 'success' ) {

        #display_success($response);
        my $id_url   = $response->{identity_url};
        my $sreg     = $response->extensionResponse('sreg');
        my $fullname = $sreg->{'fullname'};
        $email     = $sreg->{'email'};
        $authLogin = $id_url;
        $authUser  = $id_url;
        $wikiName  = $fullname || $id_url;
        $wikiName =~ s/ //g;
        
        #print STDERR "---login... SUPER: SUPERauthUser ; login: $authLogin ; wikiname: $wikiName ; cUID: $cUID ; email: $email ;\n";
    }
    else {
        #failed to auth..
        #print STDERR "--- failed to OpenID login (".$response->status.")\n";
        $authUser = $this->SUPER::login($query, $twikiSession);
    }

    $this->userLoggedIn($authLogin);

    if ( !$origurl || $origurl eq $query->url() ) {
        my $topic = $twiki->{topicName};
        my $web   = $twiki->{webName};

        $origurl = $twiki->getScriptUrl( 0, 'view', $web, $topic );
    }

    # Eat these so there's no risk of accidental passthrough
    $query->delete( 'origurl', 'username', 'password' );
    $query->delete(
        'openid.sig',        'openid.signed',
        'openid.sreg.email', 'openid.sreg.fullname',
        'openid.return_to',  'openid.assoc_handle',
        'openid.identity',   'openid.mode',
        'nonce'
    );

    # Redirect with passthrough again, so that we don't get the openid mess
    $this->redirectCgiQuery($query, $origurl);
}

1;
