# Plugin for TWiki Collaboration Platform, http://TWiki.org/
#
# Copyright (C) 2004-2005 Aurelio A. Heckert, aurium@gmail.com
# Copyright (C) 2006-2019 TWiki:TWiki.TWikiContributor
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details, published at 
# http://www.gnu.org/copyleft/gpl.html

use strict;
package TWiki::Plugins::LinkOptionsPlugin;

our $web;
our $topic;
our $user;
our $installWeb;
our $VERSION = '$Rev: 30677 (2019-10-09) $';
our $RELEASE = '2019-09-17';
our $pluginName = 'LinkOptionsPlugin';  # Name of this Plugin
our $NO_PREFS_IN_TOPIC = 1;
our $SHORTDESCRIPTION = 'Extends the _"Forced Specific Links"_ syntax ==[<nop>[URL or <nop>TopicName][Link Text][Options]]==';
our $numLastWinWithoutName;

my $LINKHEAD = qr/^(?:<noautolink>)?<a/i; # Item7840
my $debug;

# =========================
sub initPlugin
{
    ( $topic, $web, $user, $installWeb ) = @_;

    # check for Plugins.pm versions
    if( $TWiki::Plugins::VERSION < 1 ) {
        TWiki::Func::writeWarning( "Version mismatch between $pluginName and Plugins.pm" );
        return 0;
    }

    # Get plugin debug flag
    $debug = TWiki::Func::getPreferencesFlag( "\U$pluginName\E_DEBUG" );

    # Plugin correctly initialized
    TWiki::Func::writeDebug( "- TWiki::Plugins::${pluginName}::initPlugin( $web.$topic ) is OK" ) if $debug;
    $numLastWinWithoutName = 0;
    return 1;
}

# =========================
sub preRenderingHandler
{
    # do not uncomment, use $_[0], $_[1]... instead
    #my( $text, $pMap ) = @_;
    TWiki::Func::writeDebug( "- ${pluginName}::preRenderingHandler( $_[1] )" )
        if $debug;
    $_[0] =~ s/\[\[([^]\n]+)\]\[([^]\n]+)\]\[([^]]+)\]\]/&handleLinkOptions($1, $2, $3)/ge;
}

# =========================

my @preDefOptions = (
  'newwin',
  'name',
  'title',
  'class',
  'id',
  'skin'
);
my @winOptions = (
  'directories',
  'location',
  'menubar',
  'resizable',
  'scrollbars',
  'status',
  'titlebar',
  'toolbar'
);

sub handleLinkOptions
{
  my ( $link, $text, $options ) = @_;
  my %extraOpts = ();
  my $style = '';
  my $extraAtt = '';
  my @sepOpt;
  
  my $html = TWiki::Func::renderText("[[$link][$text]]");
  
  $options =~ s/win([^:|]+):([^|]+)(\||$)/$1=:$2$3/g;
  my @options = split(/\|/, $options);
  foreach my $option (@options){
    @sepOpt = split(/:/, $option);
    if ( in_array(lc($sepOpt[0]), @preDefOptions)
      || in_array(lc($sepOpt[0]), @winOptions) ){
      $extraOpts{lc($sepOpt[0])} = $sepOpt[1];
    }
    else{
      $style .= "$option; ";
    }
  }
  
  if ( $extraOpts{'skin'} ){
    if ( $html =~ m/$LINKHEAD [^>]*href="[^"]*\?[^"]*skin=.+/ ){
      $html =~ s/$LINKHEAD ([^>]*href="[^"]*\?[^"]*skin=)[^&"]*(.+)/<a $1$extraOpts{'skin'}$2/;  #"
    } else {
      if ( $html =~ m/$LINKHEAD [^>]*href="[^"]*\?.+/ ){
        $html =~ s/$LINKHEAD ([^>]*href="[^"]*\?)(.+)/<a $1skin=$extraOpts{'skin'}&$2/;  #"
      } else {
        $html =~ s/$LINKHEAD ([^>]*href="[^"]*)"(.+)/<a $1?skin=$extraOpts{'skin'}"$2/;  #"
      }
    }
  }
  
  my $URL = getByER($html, ' href="([^"]*)"', 1);
  
  if ( $extraOpts{'newwin'} ){
    if ( !$extraOpts{'name'} ){
      $numLastWinWithoutName++;
      $extraOpts{'name'} = "winNumber$numLastWinWithoutName";
    }
    my $winWidth = getByER( $extraOpts{'newwin'}, '(.+)x.+', 1);
    my $winHeight = getByER( $extraOpts{'newwin'}, '.+x(.+)', 1);
    $extraAtt .= " onclick=\"open('$URL', '". $extraOpts{'name'} ."', '";
    # The defaults for the new window:
    $extraOpts{'directories'} = 0 if ! defined $extraOpts{'directories'};
    $extraOpts{'location'} = 0    if ! defined $extraOpts{'location'};
    $extraOpts{'toolbar'} = 0     if ! defined $extraOpts{'toolbar'};
    $extraOpts{'menubar'} = 0     if ! defined $extraOpts{'menubar'};
    $extraOpts{'resizable'} = 1   if ! defined $extraOpts{'resizable'};
    $extraOpts{'scrollbars'} = 1  if ! defined $extraOpts{'scrollbars'};
    $extraOpts{'status'} = 1      if ! defined $extraOpts{'status'};
    $extraOpts{'titlebar'} = 1    if ! defined $extraOpts{'titlebar'};
    foreach my $option ( @winOptions ){
      if ( defined $extraOpts{$option} ){
        $extraAtt .= $option.'='.$extraOpts{$option}.',';
      }
    }
    $extraAtt .= "width=$winWidth,height=$winHeight";
    $extraAtt .= "'); return false;\"";
  }
  
  if ( !$extraOpts{'newwin'} && $extraOpts{'name'} ){
    $html =~ s/$LINKHEAD ([^>]*)target="[^"]*"([^>]*)>(.+)/<a $1$2>$3/;  #"
    $html =~ s/$LINKHEAD (.+)/<a target="$extraOpts{'name'}" $1/;
  }
  
  if ( $extraOpts{'title'} ){
    if ( $html =~ m/$LINKHEAD [^>]*title=.+/ ){
      $html =~ s/$LINKHEAD ([^>]*)title="[^"]*"([^>]*)>(.+)/<a $1$2 title="$extraOpts{'title'}">$3/;  #"
    } else {
      $html =~ s/$LINKHEAD ([^>]*)>(.+)/<a $1 title="$extraOpts{'title'}">$2/;
    }
  }
  if ( $extraOpts{'class'} ){
    if ( $html =~ m/$LINKHEAD [^>]*class=.+/ ){ # Item7840
      $html =~ s/$LINKHEAD ([^>]*)class="[^"]*"([^>]*)>(.+)/<a $1$2 class="$extraOpts{'class'}">$3/;  #"
    } else {
      $html =~ s/$LINKHEAD ([^>]*)>(.+)/<a $1 class="$extraOpts{'class'}">$2/;
    }
  }
  if ( $extraOpts{'id'} ){
    if ( $html =~ m/$LINKHEAD [^>]*id=.+/ ){
      $html =~ s/$LINKHEAD ([^>]*)id="[^"]*"([^>]*)>(.+)/<a $1$2 id="$extraOpts{'id'}">$3/;  #"
    } else {
      $html =~ s/$LINKHEAD ([^>]*)>(.+)/<a $1 id="$extraOpts{'id'}">$2/;
    }
  }
  
  $style = qq(style="$style") if ( $style ne '' );
  $html =~ s/<a (.+)/<a $extraAtt $style $1/;
  return $html;
}

# =========================

sub in_array {
  return if ($#_ < 1);
  my ($what, @where) = (@_);
  foreach (@where) {
    if ($_ eq $what) {
      return 1;
    }
  }
  return;
}

# =========================
no strict "refs";
sub getByER {
  my ( $str, $er, $numGrupo ) = @_;
  if ( $str =~ m/$er/ ){
    return ${$numGrupo};
  }
}

1;
