# Plugin for TWiki Enterprise Collaboration Platform, http://TWiki.org/
#
# Copyright (C) 2015 Alba Power Quality Solutions
# Copyright (C) 2015 Wave Systems Corp.
# Copyright (C) 2015-2021 Peter Thoeny, peter09[at]thoeny.org
# and TWiki Contributors. All Rights Reserved.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version. For
# more details read LICENSE in the root of this distribution.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details, published at
# http://www.gnu.org/copyleft/gpl.html
#
# As per the GPL, removal of this notice is prohibited.

package TWiki::Plugins::IfThenActionPlugin::ThenAction::SetFormField;
use base 'TWiki::Plugins::IfThenActionPlugin::ThenAction';

use strict;

# =========================
sub new {
    my $class = shift;
    my $this = $class->SUPER::new( @_ );
    $this->writeDebug( "SetFormField::new() - constructor" );
    return $this;
}

# =========================
# $this->handleAction( $web, $topic, $text, $meta, $target ) -> $message
#   * $web:    Name of source web
#   * $topic:  Name of source topic
#   * $text:   Text of source topic, possibly undef
#   * $meta:   Metadata of source topic, possibly undef
#   * $target: Target string
# return: $message: Error or normal message
#
sub handleAction {
    my ( $this, $web, $topic, $text, $meta, $target ) = @_;

    # $target is expected to be an assigment statement of form 'target = expression', such as:
    # '$children/$formfield(Status) = $formfield(Status)'
    $this->writeDebug( "SetFormField::handleAction( $web.$topic, '$target' )" );

    # split up assignment statement into target and expression text
    my $eValue = '';
    if( $target =~ s/ *\= *(.*)$// ) {
        $eValue = $1;
        $eValue = '' unless defined( $eValue );
        $eValue =~ s/ +$//;  # cut trailing spaces
    } else {
        $this->writeDebug( " - error in assignment '$target'" );
        return "ERROR: Syntax error in assignment '$target'";
    }

    my $user = TWiki::Func::getWikiName();
    if( $eValue =~ /\$formfield\([^\)]+\)$/ ) {
        # expression is of form: $formfield(...), e.g. we need to expand it

        # add source topic name to form field if missing, and expand topic and expression
        $eValue = '$topic/' . $eValue unless( $eValue =~ /\// );
        my @eItems = $this->expandTopics( $web, $topic, $meta, $eValue, 1 );
        my $eItem;
        unless( scalar @eItems && ( $eItem = $eItems[0] ) ) {
            $this->writeDebug( " - error in assignment statement '$eValue':"
                             . " Syntax error or topic not found" );
            return "ERROR: Syntax error or topic not found in assignment statement '$eValue'";
        }
        my ( $eWeb, $eTopic, $eField ) = ( $eItem->{w}, $eItem->{t}, $eItem->{a} );
        unless( $eField =~ s/ *\$formfield\(([^\)]+)\) */$1/ ) {
            $this->writeDebug( " - error in syntax of expression '$eValue'" );
            return "ERROR: Syntax error in expression '$eValue'";
        }
        undef $eValue;
        my( $eMeta, $eText ) = TWiki::Func::readTopic( $eWeb, $eTopic );
        my $permission = TWiki::Func::checkAccessPermission( 'VIEW', $user, $eText, $eTopic,
                                                             $eWeb, $eMeta );
        unless( $permission ) {
            $this->writeDebug( " - error: Topic $eWeb.$eTopic can't be viewed by $user" );
            return "ERROR: Topic $eWeb.$eTopic can't be viewed by $user";
        }
        if( $eText ) {
            my $field = $eMeta->get( 'FIELD', $eField );
            if( $field ) {
                $eValue = $field->{value};
            }
        }
        unless( defined $eValue ) {
            $this->writeDebug( " - error: Form field '$eField' does not exist in source $eWeb.$eTopic" );
            return "ERROR: Form field '$eField' does not exist in source $eWeb.$eTopic";
        }

    } else {
        # we simply use $eValue, the value of the expression (it could be an empty string)
    }

    # loop through target topic list
    my $message = '';
    foreach my $item ( $this->expandTopics( $web, $topic, $meta, $target, 1 ) ) {
        my ( $tWeb, $tTopic, $tField ) = ( $item->{w}, $item->{t}, $item->{a} );
        $this->writeDebug( "SetFormField::handleAction( $web.$topic ) : "
                         . "Update form field in '$tWeb.$tTopic'" );
        unless( $tField =~ s/ *\$formfield\(([^\)]+)\) */$1/ ) {
            $this->writeDebug( " - error in syntax of expression $tField'" );
            next;
        }

        # read, update & save topic
        my( $tMeta, $tText ) = TWiki::Func::readTopic( $tWeb, $tTopic );
        my $tValue;
        my $permission = TWiki::Func::checkAccessPermission( 'CHANGE', $user, $tText, $tTopic,
                                                             $tWeb, $tMeta );
        unless( $permission ) {
            $this->writeDebug( " - error: Target $tWeb.$tTopic can't be changed by $user" );
            next;
        }
        if( $tText ) {
            my $field = $tMeta->get( 'FIELD', $tField );
            if( $field ) {
                $tValue = $field->{value};
                $field->{value} = $eValue;
                $tMeta->putKeyed( 'FIELD', $field );
            }
        }
        unless( defined $tValue ) {
            $this->writeDebug( " - error: Form field '$tField' does not exist in target $tWeb.$tTopic" );
            next;
        }
        if( $tValue eq $eValue ) {
            $this->writeDebug( " - note: No action, target form field '$tField' in $tWeb.$tTopic"
                             . " is identical to expression value '$eValue'" );
            next;
        }
        my $err = TWiki::Func::saveTopic( $tWeb, $tTopic, $tMeta, $tText, {} );
        if( $err ) {
            $this->writeDebug( " - save error on $tWeb.$tTopic: $err" );
            $this->writeLog( "then-action: setformfield, field: $tField, topic: $tWeb.$tTopic"
                           . ", error: $err" );
            $message .= '; ' if( $message );
            $message .= "ERROR: Can't save topic '$tWeb.$tTopic', $err";
        } else {
            $this->writeLog( "then-action: setformfield, field: $tField, topic: $tWeb.$tTopic" );
        }
    }

    return $message;
}

1;
