# Plugin for TWiki Enterprise Collaboration Platform, http://TWiki.org/
#
# Copyright (C) 2015-2021 Peter Thoeny, peter09[at]thoeny.org
# and TWiki Contributors. All Rights Reserved.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version. For
# more details read LICENSE in the root of this distribution.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details, published at
# http://www.gnu.org/copyleft/gpl.html
#
# As per the GPL, removal of this notice is prohibited.

package TWiki::Plugins::IfThenActionPlugin::ThenAction::AttachSection;
use base 'TWiki::Plugins::IfThenActionPlugin::ThenAction';

use strict;

# =========================
sub new {
    my $class = shift;
    my $this = $class->SUPER::new( @_ );
    $this->writeDebug( "AttachSection::new() - constructor" );
    return $this;
}

# =========================
# $this->handleAction( $web, $topic, $text, $meta, $target ) -> $message
#   * $web:    Name of source web
#   * $topic:  Name of source topic
#   * $text:   Text of source topic, possibly undef
#   * $meta:   Metadata of source topic, possibly undef
#   * $target: Target string
# return: $message: Error or normal message
#
sub handleAction {
    my ( $this, $web, $topic, $text, $meta, $target ) = @_;

    # Target syntax example: $topic/*.css
    $this->writeDebug( "AttachSection::handleAction( $web.$topic, '$target' )" );

    # loop through topic list
    my $message = '';
    my $user = TWiki::Func::getWikiName();
    foreach my $item ( $this->expandTopics( $web, $topic, $meta, $target, 1 ) ) {
        my ( $tWeb, $tTopic, $tSection ) = ( $item->{w}, $item->{t}, $item->{a} );
        $tSection =~ s/\*/.*/;
        $tSection =~ s/\?/./;

        unless( $tSection ) {
            $this->writeDebug( " - error: Section name is missing in target '$target'" );
            return "ERROR: Section name is missing in target '$target'";
        }
        my( $tMeta, $tText ) = TWiki::Func::readTopic( $tWeb, $tTopic );
        my $permission = TWiki::Func::checkAccessPermission( 'VIEW', $user, $tText, $tTopic,
                                                             $tWeb, $tMeta );
        unless( $permission ) {
            $this->writeDebug( " - error: Target topic $tWeb.$tTopic can't be viewed by $user" );
            return "ERROR: Target topic $tWeb.$tTopic can't be viewed by $user";
        }
        unless( $tText ) {
            $this->writeDebug( " - error: Target topic $tWeb.$tTopic is empty" );
            return "ERROR: Target topic $tWeb.$tTopic is empty";
        }
        # Quick & dirty way to identify sections.
        # Limitation: Sections split up into multiple parts are not supported
        my %s;
        $text =~ s/\%STARTSECTION\{[ "']*([^ "'}]+)[^}]*}\%(.*?)\%ENDSECTION\{[ "']*(\1)[^}]*}\%/$s{$1} = $2/geos;
        my $c = 0;
        foreach my $section ( keys %s ) {
            unless( $section =~ /^$tSection$/ ) {
                $this->writeDebug( "AttachSection::handleAction( $web.$topic ) : "
                                 . "Skipping section $section'" );
                next;
            }
            $this->writeDebug( "AttachSection::handleAction( $web.$topic ) : "
                         . "Attach section '$section' as file '$section'" );
            # attach content of section using section name $section
            $this->_saveAsAttachment( $tWeb, $tTopic, $section, $s{$section} );
            $this->writeLog( "then-action: attachsection, topic: $tWeb.$tTopic, section: $section" );
            $c++
        }
        unless( $c ) {
            $this->writeDebug( " - error: No section match in target topic $tWeb.$tTopic" );
            return "ERROR: No section match in target topic $tWeb.$tTopic";
        }
    }

    return $message;
}

sub _saveAsAttachment {
    my ( $this, $web, $topic, $fileName, $text ) = @_;

    $fileName =~ s/[^0-9a-zA-Z\.-_]//g; # clean filename
    return unless $fileName;
    my $workArea = TWiki::Func::getWorkArea( 'IfThenActionPlugin' );
    my $tempName = $workArea . '/tmp' . int( rand( 1000000 ) );
    my $content = TWiki::Func::expandCommonVariables( $text, $topic, $web );
    $content =~ s/<nop>//goi;
    $content =~ s/^[\n\r]+//s;
    $content =~ s/[\n\r]*$/\n/s;
    TWiki::Func::saveFile( $tempName, $content );
    return unless( -e $tempName );
    my @stats = stat $tempName;
    my $fileSize = $stats[7];
    my $fileDate = $stats[9];
    my $twikiWeb = $TWiki::cfg{SystemWebName};
    TWiki::Func::saveAttachment( $web, $topic, $fileName,
      {
        file => $tempName,
        filedate => $fileDate,
        filesize => $fileSize,
        filepath => $fileName,
        comment => "Auto-attached by $twikiWeb.IfThenActionPlugin",
        hide => 1,
      });
    unlink( $tempName );
}

1;
