# Plugin for TWiki Enterprise Collaboration Platform, http://TWiki.org/
#
# Copyright (C) 2015 Alba Power Quality Solutions
# Copyright (C) 2015 Wave Systems Corp.
# Copyright (C) 2015-2021 Peter Thoeny, peter09[at]thoeny.org
# and TWiki Contributors. All Rights Reserved.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version. For
# more details read LICENSE in the root of this distribution.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details, published at
# http://www.gnu.org/copyleft/gpl.html
#
# As per the GPL, removal of this notice is prohibited.

package TWiki::Plugins::IfThenActionPlugin::ThenAction;

use strict;

our $debug = $TWiki::cfg{Plugins}{IfThenActionPlugin}{Debug} || 0;

# =========================
sub new {
    my $class = shift;
    my %attrs = @_;

    $attrs{name} ||= 'unknown';

    return bless( \%attrs, $class );
}

# =========================
# $this->handleAction( $web, $topic, $text, $meta, $target ) -> $message
#   * $web:    Name of source web
#   * $topic:  Name of source topic
#   * $text:   Text of source topic, possibly undef
#   * $meta:   Metadata of source topic, possibly undef
#   * $target: Target string
# return: $message: Error or normal message
#
sub handleAction {
    ##my ( $this, $web, $topic, $text, $meta, $target ) = @_;
    return '';
}

# =========================
# The following are helper methods, do not overload

# =========================
# $this->expandTopics( $sWeb, $sTopic, $sMeta, $target, $hasAccessor ) -> @webTopics
#   * $sWeb:   Name of source web, used to normalize Web.TopicName
#   * $sTopic: Name of source topic, used to expand tokens
#   * $sMeta:  Metadata of source topic, used to expand tokens
#   * $target: Comma-space list of items with topics & tokens. Topics are normalized to
#     Web.TopicName. Non-existing topics are removed. These tokens are expanded:
#      * $topic: name of source topic
#      * $parents: parent trail, up to 64 parents up; item is removed if none
#      * $parents(5): parent trail, up to 5 parents up; item is removed if none
#      * $parent: topic parent; item is removed if none
#      * $parent(5): the 5th parent up in the parent trail; item is removed if none
#      * $children: expands to all direct children; item is removed if none
#      * $children(5): expands to all children recursively 5 levels deep; item is removed if none
#      * $children(5 CID-*): expands to all children starting with CID-* recursively 5 levels deep
#     Example target: Web.SomeTopic, Web.NotExist, Web.$parent, Web.$children
#     Example output: Web.SomeTopic, Web.ItsParent, Web.ItsChild1, Web.ItsChild2, Web.ItsChild3
#   * $hasAccessor: '1' if items may contain accessors; accessors are extracted and returned.
#     Example: Web.SomeTopic/formfield(Status)
#   * Return: @webTopics: Array of hash references
#     Example: ( { w => 'web1', t => 'topic1', a => 'formfield(Status)' },
#                { w => 'web2', t => 'topic2', a => '' }, ... )
#
sub expandTopics {
    my ( $this, $sWeb, $sTopic, $sMeta, $target, $hasAccessor ) = @_;

    $this->writeDebug( "expandTokens( $sWeb.$sTopic, $target )" );
    my @webTopics = ();
    foreach my $item ( split( /, */, $target ) ) {
        my $accessor = '';
        if( $hasAccessor && $item =~ s/(.*)\/(.*)$/$1/ ) {
            $accessor = $2;
        }

        # do Web.TopicName normalization, can't use TWiki::Func::normalizeWebTopicName
        # yet because of tokens
        my ( $web, $topic ) = ( $sWeb, $item );
        if( $topic =~ s/^(.*)\.// ) {
            $web = $1;
        }
        next unless( $web && $topic );

        # handle all tokens; result can be a single topic or a space delimited list of topics
        my $topics = $topic;
        $topics =~ s/\$topic/$sTopic/go;
        $topics =~ s/\$parents\( *([0-9]+) *\)/$this->_getParents( $sWeb, $sTopic, $sMeta, $1, 0 )/geo;
        $topics =~ s/\$parents/$this->_getParents( $sWeb, $sTopic, $sMeta, 64, 0 )/geo;
        $topics =~ s/\$parent\( *([0-9]+) *\)/$this->_getParents( $sWeb, $sTopic, $sMeta, $1, 1 )/geo;
        $topics =~ s/\$parent/$this->_getParents( $sWeb, $sTopic, $sMeta, 1, 1 )/geo;
        $topics =~ s/\$children\( *([0-9]+)([^\)]*)\)/$this->_getChildren( $sWeb, $sTopic, $1, $2, 1 )/geo;
        $topics =~ s/\$children/$this->_getChildren( $sWeb, $sTopic, 1, '', 1 )/geo;
        $topics =~ s/ +$//go;
        $topics =~ s/^ +//go;
        $this->writeDebug( " - topics: $topics" );
        next unless( $topics );

        foreach my $topic ( split( / +/, $topics ) ) {

            # now we are ready to normalize Web.TopicName
            ( $web, $topic ) = TWiki::Func::normalizeWebTopicName( $web, $topic );
            $web   =~ s/$TWiki::cfg{NameFilter}//go;
            $topic =~ s/$TWiki::cfg{NameFilter}//go;
            $web = _untaintChecked($web);
            $topic = _untaintChecked($topic);
            next unless( TWiki::Func::topicExists( $web, $topic ) );

            push( @webTopics, { w => $web, t => $topic, a => $accessor } );
        }
    }
    return @webTopics;
}

# =========================
sub _getParents {
    my ( $this, $web, $topic, $meta, $level, $onlyOne ) = @_;

    $this->writeDebug( "_getParents( $web.$topic, $level, $onlyOne )" );
    unless( $meta ) {
        ( $meta ) = TWiki::Func::readTopic( $web, $topic );
        return '' unless( $meta );
    }
    my $parent = $meta->getParent();
    return '' unless( $parent );
    my ( $pWeb, $pTopic ) = TWiki::Func::normalizeWebTopicName( $web, $parent );
    return '' unless( $pWeb eq $web ); # ignore parent across webs

    $level = 64 if( $level > 64 );
    if( $pTopic && --$level > 0 ) {
        # recursively get grand-parent
        $topic = $pTopic;
        $pTopic = $this->_getParents( $web, $topic, undef, $level, $onlyOne );
        $pTopic .= $topic unless( $onlyOne );
    }
    return " $pTopic "; # use space as separator
}

# =========================
sub _getChildren {
    my ( $this, $web, $topic, $level, $topicFilter, $firstTime ) = @_;

    $topicFilter ||= '';
    $this->writeDebug( "_getChildren( $web.$topic, $level, $topicFilter, $firstTime )" );
    if( $firstTime ) {
        my @topics = TWiki::Func::getTopicList( $web );
        $topicFilter =~ s/ +//g;
        $topicFilter =~ s/\*/.*/g;
        $topicFilter =~ s/\?/./g;
        if( $topicFilter ) {
            @topics = grep{ /^$topicFilter$/ } @topics;
        }
        if( $level <= 1 ) {
            # special case, search just for direct children
            my $search = "[%]META:TOPICPARENT\{name=\"$topic\"\}";
            my $result = TWiki::Func::searchInWebContent( $search, $web, \@topics,
                { type => 'regex', files_without_match => 1 } );
            my $children = ' ' . join( ' ', keys %$result ) . ' ';
            return $children;

        } else {
            # search all topic to create a topic/parent hash. This is slower than the
            # special case, but faster than a recursive search.
            my $search = "[%]META:TOPICPARENT\{name=\"[^\"]+\"\}";
            my $result = TWiki::Func::searchInWebContent( $search, $web, \@topics,
                { type => 'regex' } );
            delete $this->{parents} if( $this->{parents} );
            foreach my $t ( keys %$result ) {
                my $line = ${ $result->{$t} }[0]; # use only the first hit for parent
                if( $line =~ /[%]META:TOPICPARENT\{name=\"([^\"]+)\"\}/ ) {
                    my $p = $1;
                    next if( $p =~ /\./ ); # ignore parent across webs
                    $this->{parents}{$t} = $p;
                }
            }
        }
    }
    return '' unless( $this->{parents} ); # search on first run did not find any topics with parent

    my $children = '';
    $level = 64 if( $level > 64 );
    $level--;
    foreach my $k ( keys %{ $this->{parents} } ) {
        if( $this->{parents}{$k} eq $topic ) {
            $children .= " $k";
            if( $level > 0 ) {
                # recursively get grand-children
                $children .= $this->_getChildren( $web, $k, $level, '', 0 );
            }
        }
    }
    return $children;
}

#==========================
sub _untaintChecked {
    my( $text ) = @_;
    $text = $1 if( $text =~ /^(.*)$/ );
    return $text;
}

# =========================
sub writeLog {
    my ( $this, $msg ) = @_;
    return unless( $TWiki::Plugins::VERSION >= 1.4 );
    TWiki::Func::writeLog( 'ifthenaction', $msg );
}

# =========================
sub writeDebug {
    my ( $this, $msg ) = @_;
    return unless( $debug );
    TWiki::Func::writeDebug( "- IfThenActionPlugin::TheAction::$msg" );
}

# =========================
sub writeError {
    my ( $this, $msg ) = @_;
    print STDERR "ERROR TWiki IfThenActionPlugin::ThenAction::$msg\n";
}

1;
