# Plugin for TWiki Enterprise Collaboration Platform, http://TWiki.org/
#
# Copyright (C) 2015 Alba Power Quality Solutions
# Copyright (C) 2015 Wave Systems Corp.
# Copyright (C) 2015-2021 Peter Thoeny, peter09[at]thoeny.org
# and TWiki Contributors. All Rights Reserved.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version. For
# more details read LICENSE in the root of this distribution.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details, published at
# http://www.gnu.org/copyleft/gpl.html
#
# As per the GPL, removal of this notice is prohibited.

package TWiki::Plugins::IfThenActionPlugin::Core;

our $debug = $TWiki::cfg{Plugins}{IfThenActionPlugin}{Debug} || 0;

# =========================
sub new {
    my ( $class ) = @_;

    # discover TWiki bin dir
    my $binDir = $ENV{SCRIPT_FILENAME} || '';
    $binDir =~ s|(.*)[\\/]+.*|$1|;       # cut off script to get name of bin dir
    unless( $binDir )  {
        # last resort to discover bin dir
        require Cwd;
        import Cwd qw( cwd );
        $binDir = cwd();
    }
    my $this = {
          RulesTopic  => $TWiki::cfg{Plugins}{IfThenActionPlugin}{RulesTopic}
                       || 'Main.IfThenActionRules',
          BinDir      => _untaintChecked( $binDir ),
          WorkDir     => TWiki::Func::getWorkArea( 'IfThenActionPlugin' ),
        };

    my $disabled = $TWiki::cfg{Plugins}{IfThenActionPlugin}{DisableIfActions} || '';
    my @ifActions = ( 'action' );
    push( @ifActions, 'register' ) unless( $disabled =~ /\b(register)\b/ );
    push( @ifActions, 'save' )     unless( $disabled =~ /\b(save)\b/ );
    push( @ifActions, 'upload' )   unless( $disabled =~ /\b(upload)\b/ );
    push( @ifActions, 'view' )     unless( $disabled =~ /\b(view)\b/ );
    $this->{IfActions} = join( ', ', @ifActions );

    bless( $this, $class );
    _writeDebug( "new() - constructor" );

    return $this;
}

# =========================
sub VarIFTHEN {
    my ( $this, $params, $topic, $web, $meta ) = @_;

    _writeDebug( "VarIFTHEN( " . $params->stringify() . " )" );
    my $action = $params->{_DEFAULT};

    if( $action eq 'if-actions' ) {
        return $this->{IfActions};

    } elsif( $action eq 'then-actions' ) {
        my $thenActions = $this->_loadThenActions();
        return join( ', ', sort keys %{$thenActions} )

    } elsif( $action eq 'action' ) {
        $topic = $params->{topic} if( $params->{topic} );
        ( $web, $topic ) = TWiki::Func::normalizeWebTopicName( $web, $topic );
        return $this->handleIfThenAction( 'action', $web, $topic, undef, $meta );

    } elsif( $action eq '' ) {
        # return empty string if no script name given, useful for interactive apps
        return '';

    } else {
        return "IFTHEN ERROR: Unknown action '$action'";
    }
}

# =========================
sub handleIfThenAction {
    my $this     = $_[0];
    my $ifAction = $_[1];
    my $web      = $_[2];
    my $topic    = $_[3];
    ##my $text   = $_[4];
    ##my $meta   = $_[5];

    _writeDebug( "handleIfThenAction( '$ifAction' on '$web.$topic' )" ) if( $debug );

    # disable recursive if-then action, e.g. we do not want one save action to trigger
    # another save action
    return '' if( $this->{disableSelf} );    
    $this->{disableSelf} = 1;
    my $message = '';

    # load if-then-action rules, and loop through them
    my $ifThenRules = $this->_loadIfThenRules();
    my $thenActions;  # then-actions are loaded later if needed 
    foreach my $rule ( @{ $ifThenRules } ) {
        next unless( $rule->{'if'} eq $ifAction );  # if-action does not apply to us

        # test then-action
        my $thenAction = $rule->{then};
        $thenActions = $this->_loadThenActions() unless( $thenActions );
        next unless( defined $thenActions->{$thenAction} );  # then-action does not exist

        # prepare source
        my $source = $rule->{source};
        if( $source =~ /\%[A-Za-z]/ ) {
            # detected potential TWiki variable, so expand it
            $source = TWiki::Func::expandCommonVariables( $source, $topic, $web, $_[5] );
        }
        next unless( $source );  # bail out if no source

        # source can be a list of form "Web.Topic1, Web.Topic2"
        foreach my $item ( split( /, */, $source ) ) {
            my ( $sWeb, $sTopic ) = TWiki::Func::normalizeWebTopicName( $web, $item );
            if( $sTopic =~ /[\*\?]/ ) {
                # wildcard detected, do regex compare
                $sTopic =~ s/\*/.*/g;
                $sTopic =~ s/\?/./g;
                next unless( "$web.$topic" =~ /^$sWeb\.$sTopic$/ );
            } else {
                # do web.topic string compare
                $sWeb   =~ s/$TWiki::cfg{NameFilter}//go;
                $sTopic =~ s/$TWiki::cfg{NameFilter}//go;
                next unless( "$web.$topic" eq "$sWeb.$sTopic" );
            }

            # prepare target
            my $target = $rule->{target};
            if( $target =~ /\%[A-Za-z]/ ) {
                # detected potential TWiki variable, so expand it
                $target = TWiki::Func::expandCommonVariables( $target, $topic, $web, $_[5] );
            }

            # do then-action
            _writeDebug( "handleIfThenAction rule: if=>'$ifAction',"
                       . " source=>'$sWeb.$sTopic', then=>'$thenAction',"
                       . " target=>'$target'" ) if( $debug );
            my $msg = $thenActions->{$thenAction}->handleAction( $web, $topic, $_[4], $_[5], $target );
            if( $msg ) {
                $message .= '; ' if( $message );
                $message .= $msg;
            }
        }
    }
    undef $this->{disableSelf};
    return $message;
}

#==========================
sub _loadIfThenRules {
    my ( $this ) = @_;

    unless( $this->{IfThenRules} ) {
        my @rules = ();
        my ( $web, $topic ) = TWiki::Func::normalizeWebTopicName( '', $this->{RulesTopic} );
        my $text = TWiki::Func::readTopicText( $web, $topic, undef, 1 );
        foreach my $line ( split( /([\r\n]+)/, $text ) ) {
            # Example rule:
            # | *If* | *Source* | *Then* | *Target* | *Comment* |
            # | save | Sandbox.IfThenActionTest | view | Sandbox.IfThenActionView | comment |
            if( $line =~ /^ *\| * ([a-z]+) *\| *([^\|]+)\| *([^\|]+)\| *([^\|]+)\|/ ) {
                my $if     = $1;
                my $source = $2;
                my $then   = $3;
                my $target = $4;
                $source =~ s/ *$//;
                $then   =~ s/ *$//;
                $target =~ s/ *$//;
                my $aRule = { 'if' => $if, source => $source, then => $then, target => $target };
                push( @rules, $aRule );
            }
        }
        $this->{IfThenRules} = \@rules;
    }
    return $this->{IfThenRules};
}

#==========================
sub _loadThenActions {
    my ( $this ) = @_;

    unless( $this->{ThenActions} ) {
        # discover lib dir via twiki/lib/TWiki.pm
        foreach my $dir ( @INC ) {
            if( -e "$dir/TWiki.pm" ) {
                $this->{LibDir} = _untaintChecked( $dir );
                last;
            }
        }

        # load all ThenAction modules
        my $disabled = $TWiki::cfg{Plugins}{IfThenActionPlugin}{DisableThenActions} || '';
        $this->{ThenActions} = {};
        my $dir = $this->{LibDir} . '/TWiki/Plugins/IfThenActionPlugin/ThenAction';
        if ( opendir( DIR, $dir ) ) {
            foreach my $module ( map { s/\.pm$//; $_ } grep{ /\.pm$/ } readdir( DIR ) ) {
                my $name = lc( $module );
                next if( $disabled =~ /\b$name\b/ );
                $module = 'TWiki::Plugins::IfThenActionPlugin::ThenAction::'
                        . _untaintChecked( $module );
                eval 'require ' . $module;
                if( $@ ) {
                    # Error loading module
                    _writeDebug( "_loadThenActions: Error loading $module" );
                    _writeError( "_loadThenActions: Error loading $module" );
                    next;
                 }
                 $this->{ThenActions}{$name} = $module->new(
                     name  => $name,
                     core  => $this,
                   );
            }
            closedir( DIR );
        }
    }
    return $this->{ThenActions};
}

#==========================
sub _untaintChecked {
    my( $text ) = @_;
    $text = $1 if( $text =~ /^(.*)$/ );
    return $text;
}

# =========================
sub _writeDebug
{
    my ( $msg ) = @_;
    return unless( $debug );
    TWiki::Func::writeDebug( "- IfThenActionPlugin::Core::$msg" );
}

# =========================
sub _writeError
{
    my ( $msg ) = @_;
    print STDERR "ERROR TWiki IfThenActionPlugin::Core::$msg\n";
}

# =========================
1;
