# Plugin for TWiki Enterprise Collaboration Platform, http://TWiki.org/
#
# Copyright (C) 2015 Alba Power Quality Solutions
# Copyright (C) 2015 Wave Systems Corp.
# Copyright (C) 2015-2021 Peter Thoeny, peter[at]thoeny.org 
# Copyright (C) 2015-2021 TWiki Contributors. All Rights Reserved.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version. For
# more details read LICENSE in the root of this distribution.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details, published at
# http://www.gnu.org/copyleft/gpl.html
#
# As per the GPL, removal of this notice is prohibited.

package TWiki::Plugins::IfThenActionPlugin;

# =========================
our $VERSION = '$Rev: 31010 (2021-03-25) $';
our $RELEASE = '2021-03-25';
our $SHORTDESCRIPTION = 'Execute if-then actions on topic view, save, file upload, and '
                      . 'user registration, with rules defined in a topic';
our $NO_PREFS_IN_TOPIC = 1;

# =========================
our $debug = $TWiki::cfg{Plugins}{IfThenActionPlugin}{Debug} || 0;
our $disableIfActions = $TWiki::cfg{Plugins}{IfThenActionPlugin}{DisableIfActions} || '';
our $web;
our $topic;
our $core;

# =========================
sub initPlugin {
    ( $topic, $web ) = @_;

    # check for Plugins.pm versions
    if( $TWiki::Plugins::VERSION < 1.2 ) {
        TWiki::Func::writeWarning( "Version mismatch between IfThenActionPlugin and Plugins.pm" );
        return 0;
    }

    TWiki::Func::registerTagHandler( 'IFTHEN', \&_IFTHEN );

    # Plugin correctly initialized
    TWiki::Func::writeDebug( "- TWiki::Plugins::IfThenActionPlugin::initPlugin( $web.$topic ) is OK" )
      if( $debug );

    return 1;
}

# =========================
sub _IFTHEN {
    my ( $session, $params, $theTopic, $theWeb, $meta ) = @_;

    # Lazy loading, e.g. compile core module only when required
    unless( $core ) {
        require TWiki::Plugins::IfThenActionPlugin::Core;
        $core = new TWiki::Plugins::IfThenActionPlugin::Core();
    }
    return $core->VarIFTHEN( $params, $theTopic, $theWeb, $meta );
}

# =========================
sub afterSaveHandler {
    # do not uncomment, use $_[0], $_[1]... instead
    ### my ( $text, $topic, $web, $error, $meta ) = @_;

    return if( $_[3] ); # cancel action in case of save error
    return if( $disableIfActions =~ /\b(save)\b/ );

    # Lazy loading, e.g. compile core module only when required
    unless( $core ) {
        require TWiki::Plugins::IfThenActionPlugin::Core;
        $core = new TWiki::Plugins::IfThenActionPlugin::Core();
    }
    $core->handleIfThenAction( 'save', $_[2], $_[1], $_[0], $_[4] );
    return;
}

# =========================
sub afterAttachmentSaveHandler {
    # do not uncomment, use $_[0], $_[1]... instead
    ###   my( $attrHashRef, $topic, $web, $error, $meta ) = @_;

    return if( $_[3] ); # cancel action in case of upload error
    return if( $disableIfActions =~ /\b(upload)\b/ );

    # Lazy loading, e.g. compile core module only when required
    unless( $core ) {
        require TWiki::Plugins::IfThenActionPlugin::Core;
        $core = new TWiki::Plugins::IfThenActionPlugin::Core();
    }
    $core->handleIfThenAction( 'upload', $_[2], $_[1], undef, $_[4] );
    return;
}

# =========================
sub registrationHandler {
    my ( $web, $wikiName ) = @_;

    return if( $disableIfActions =~ /\b(register)\b/ );

    # Work around TWkibug:Item7642 in TWiki::UI::Register, fixed 2015-04-15
    $web ||= $TWiki::cfg{UsersWebName}; 

    # Lazy loading, e.g. compile core module only when required
    unless( $core ) {
        require TWiki::Plugins::IfThenActionPlugin::Core;
        $core = new TWiki::Plugins::IfThenActionPlugin::Core();
    }
    $core->handleIfThenAction( 'register', $web, $wikiName, undef, undef );
    return;
}

# =========================
sub completePageHandler {
    #my($html, $httpHeaders) = @_;

    return unless( TWiki::Func::getContext()->{view} );
    if( $disableIfActions =~ /\b(view)\b/ ) {
        # one more check before disabling 'view' if-action
        my $query = TWiki::Func::getCgiQuery();
        return unless( $query );
        my $ifta = $query->param( 'ifta' ) || '';
        # ifta=view is an undocumented URL paramater to cancel the disabling of the if-action
        return unless( $ifta =~ /^view\b/ );
    }

    # Lazy loading, e.g. compile core module only when required
    unless( $core ) {
        require TWiki::Plugins::IfThenActionPlugin::Core;
        $core = new TWiki::Plugins::IfThenActionPlugin::Core();
    }
    # Use web and topic from init time
    $core->handleIfThenAction( 'view', $web, $topic, undef, undef );
    return;
}

# =========================
1;
