# Plugin for TWiki Enterprise Collaboration Platform, http://TWiki.org/
#
# Copyright (C) 2014 Alba Power Quality Solutions
# Copyright (C) 2014-2016 Peter Thoeny, peter[at]thoeny.org 
# Copyright (C) 2014-2016 TWiki Contributors. All Rights Reserved.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 3
# of the License, or (at your option) any later version. For
# more details read LICENSE in the root of this distribution.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details, published at
# http://www.gnu.org/copyleft/gpl.html
#
# As per the GPL, removal of this notice is prohibited.

package TWiki::Plugins::ExecutePlugin::Core;

our $debug = $TWiki::cfg{Plugins}{ExecutePlugin}{Debug} || 0;

# =========================
sub new {
    my ( $class, $debug ) = @_;

    my $this = {
          ScriptFilter => $TWiki::cfg{Plugins}{ExecutePlugin}{ScriptFilter}
                       || '[^a-zA-Z0-9_\\-\\+!\\:\\., /\\(\\)\\@]',
          PerlFilter   => $TWiki::cfg{Plugins}{ExecutePlugin}{PerlFilter}
                       || '[^a-zA-Z0-9_\\-\\+!\\:\\., /\\(\\)\\@]',
          Scripts      => $TWiki::cfg{Plugins}{ExecutePlugin}{Scripts}
                       || [],
        };
    bless( $this, $class );
    _writeDebug( "new() - constructor" );

    return $this;
}

# =========================
sub VarEXECUTE
{
    my ( $this, $params ) = @_;

    _writeDebug( "VarEXECUTE( " . $params->stringify() . " )" );
    my $name = $params->{_DEFAULT};
    # return empty string if no script name given, useful for interactive apps
    return '' unless( $name );

    foreach my $scriptDef ( @{$this->{Scripts}} ) {
        if( $scriptDef->{name} eq $name ) {
            my $command = $scriptDef->{command} || '';
            return "EXECUTE ERROR: Undefined command for script named '$name'" unless( $command );
            my $type = $scriptDef->{type} || 'script';
            my $text = '';
            if( $type eq 'script' ) {
                my $filter = $scriptDef->{filter} || $this->{ScriptFilter};
                $text = $this->_executeScript( $name, $command, $filter, $params );
            } elsif( $type eq 'perl' ) {
                my $filter = $scriptDef->{filter} || $this->{PerlFilter};
                $text = $this->_executePerl( $name, $command, $filter, $params );
            } else {
                return "EXECUTE ERROR: Unknown script type '$type' for script named '$name'";
            }
            if( $params->{format} ) {
                my $format = TWiki::Func::decodeFormatTokens( $params->{format} );
                $text = join( "\n",
                    map{
                        my $line = $format;
                        $line =~ s/\$text/$_/g;
                        $line;
                    }
                    split( /\n\r?/, $text )
                  );
            }
            if( $params->{newline} ) {
                my $newline = TWiki::Func::decodeFormatTokens( $params->{newline} );
                $text =~ s/\n*$//g; # cut last newline
                $text =~ s/\n/$newline/g;
            }
            return $text;
        }
    }
    return "EXECUTE ERROR: Script named '$name' not found";
}

# =========================
sub _executeScript
{
    my ( $this, $name, $command, $filter, $params ) = @_;

    _writeDebug( "_executeScript('$name', '$command', '$filter')" ) if( $debug );
    $command =~ s/\%([a-zA-Z][a-zA-Z0-9]*)\%/$this->_substituteParam( $1, $filter, $params )/ge;
    $command =~ /^(.*)$/;
    $command = $1; # untaint
    _writeDebug( "_executeScript - command: $command" ) if( $debug );
    my $text = `$command`;
    return $text;
}

# =========================
sub _executePerl
{
    my ( $this, $name, $command, $filter, $params ) = @_;
    
    _writeDebug( "_executePerl('$name', '$command', '$filter')" );
    $command =~ s/\%([a-zA-Z][a-zA-Z0-9]*)\%/$this->_substituteParam( $1, $filter, $params )/ge;
    $command =~ /^(.*)$/; 
    $command = $1; # untaint
    _writeDebug( "_executePerl - command: $command" ) if( $debug );
    my $text = eval $command;
    unless( defined $text ) {
        $text = "EXECUTE ERROR: $@";
    }
    return $text;
}

# =========================
sub _substituteParam
{
    my ( $this, $name, $filter, $params ) = @_;
    my $text = $params->{$name};
    if( defined $text ) {
        $text =~ s/$filter//g; # apply filter
        _writeDebug( "_substituteParam('$name') - return: $text" ) if( $debug );
        return $text;
    }
    $text = '%' . $name . '%'; # restore unexpanded parameter if not found
    _writeDebug( "_substituteParam('$name') - not found, return: $text" ) if( $debug );
    return $text;
}

# =========================
sub _writeDebug
{
    my ( $msg ) = @_;
    return unless( $debug );
    TWiki::Func::writeDebug( "- ExecutePlugin::Core::$msg" );
}

# =========================
1;
