use strict;
use warnings;

package TWiki::Plugins::EmbeddedJSPlugin::Func;
use base 'Exporter';

require TWiki::Func;

use EJS::Template;
use EJS::Template::Util;
use File::Basename;
use HTML::Entities;
use IO::String;
use JSON;
use Scalar::Util qw(looks_like_number refaddr);
use Text::CSV;

our @EJS_ROOT_FUNCTIONS = qw(
    print
    println
);

our @EJS_FUNCTIONS = qw(
    requireTopic
    findWebs
    findTopics
    webExists
    topicExists
    createWeb
    moveWeb
    removeWeb
    readTopic
    saveTopic
    moveTopic
    removeTopic
    getFormName
    setFormName
    getFormField
    setFormField
    getFormFields
    checkPermission
    getRevisionInfo
    getRevisionAtTime
    getEditLock
    acquireEditLock
    releaseEditLock
    getAttachmentList
    attachmentExists
    readAttachment
    saveAttachment
    moveAttachment
    removeAttachment
    getMethod
    isPost
    getParam
    setParam
    getUserName
    getWikiName
    getWikiUserName
    getVariable
    setVariable
    expandVariables
    expandAutoInc
    formatText
    isTrue
    normalizeWebTopicName
    escapeHTML
    unescapeHTML
    escapeQuote
    unescapeQuote
    escapeTable
    unescapeTable
    parseTable
    formatTable
    parseCSV
    formatCSV
    getExternalResource
    postExternalResource
    captureOutput
    getJavaScriptEngine
    getEJSConfig
    executeEJS
);

our @EXPORT_OK = @EJS_FUNCTIONS;
our %EXPORT_TAGS = (all => \@EXPORT_OK);

our $CONTEXT = {};
our $CAPTURE;

my $alarmAlreadySet = 0;

sub _executeEJS {
    my ($textRef, $web, $topic, $meta, $ejsConfig) = @_;

    # If there are no EJS tags at all, do nothing.
    if ($$textRef !~ /<%/) {
        return;
    }

    eval {
        my $ejs = EJS::Template->new(engine => $ejsConfig->{jsEngine});
        my $adapter = $ejs->executor->adapter;
        my $engine = $adapter->engine;

        local $CONTEXT = {
            adapter        => $adapter,
            engine         => $engine,
            config         => $ejsConfig,
            web            => $web,
            topic          => $topic,
            meta           => $meta,
            requiredTopics => {},
        };

        my $variables = _initializeVariables();

        my $timeout = $ejsConfig->{timeout};
        local $SIG{ALRM} = sub {die "EJS took too long (timeout: $timeout sec.)\n"};

        my $newAlarm = 0;

        if (!$alarmAlreadySet) {
            $alarmAlreadySet = 1;
            $newAlarm = 1;
            alarm $timeout;
        }

        eval {
            EJS::Template->process($textRef, $variables, $textRef);
        };

        if ($newAlarm) {
            alarm 0;
            $alarmAlreadySet = 0;
        }
    };

    if ($@) {
        $$textRef .= "\n\n".'%RED%'."EJS: $@".'%ENDCOLOR%';
    }
}

sub _initializeVariables {
    my ($class) = @_;
    my $isJE = $CONTEXT->{engine}->isa('JE');

    my $variables = {
        JSON => {
            stringify => _wrapFunction(\&TWiki::Plugins::EmbeddedJSPlugin::Func::JSON_stringify, $isJE),
            parse     => _wrapFunction(\&TWiki::Plugins::EmbeddedJSPlugin::Func::JSON_parse, $isJE),
        },
        console => {
            log   => _wrapFunction(\&TWiki::Plugins::EmbeddedJSPlugin::Func::console_log, $isJE),
            debug => _wrapFunction(\&TWiki::Plugins::EmbeddedJSPlugin::Func::console_debug, $isJE),
            info  => _wrapFunction(\&TWiki::Plugins::EmbeddedJSPlugin::Func::console_info, $isJE),
            warn  => _wrapFunction(\&TWiki::Plugins::EmbeddedJSPlugin::Func::console_warn, $isJE),
            error => _wrapFunction(\&TWiki::Plugins::EmbeddedJSPlugin::Func::console_error, $isJE),
        },
    };

    my $namespace = _makeNamespace($variables, $CONTEXT->{config}{namespace});

    no strict 'refs';
    for my $name (@EJS_ROOT_FUNCTIONS) {
        $variables->{$name} = _wrapFunction(\&{'TWiki::Plugins::EmbeddedJSPlugin::Func::'.$name}, $isJE);
    }
    for my $name (@EJS_FUNCTIONS) {
        $namespace->{$name} = _wrapFunction(\&{'TWiki::Plugins::EmbeddedJSPlugin::Func::'.$name}, $isJE);
    }
    use strict 'refs';

    return $variables;
}

sub _makeNamespace {
    my ($root, $spec) = @_;
    return $root unless defined $spec;

    my $namespace = $root;
    $spec =~ s/^\s+|\s+$//g;

    for my $name (split /\s*\.\s*/, $spec) {
        if ($name =~ /^[A-Za-z_\$][A-Za-z_\$0-9]*$/) {
            $namespace = ($namespace->{$name} ||= {});

            if (ref($namespace) ne 'HASH') {
                die "Invalid namespace (non-object): $spec\n";
            }
        } else {
            die "Invalid namespace (unknown characters): $spec\n";
        }
    }

    return $namespace;
}

sub _wrapFunction {
    my ($func, $isJE) = @_;

    if ($isJE) {
        return sub {
            my $ret = eval {$func->(map {_je2perl($_)} @_)};
            die JE::Object::Error->new($CONTEXT->{engine}->global(), _perl2je($@)) if $@;
            return _perl2je($ret);
        };
    } else {
        return sub {
            my $ret = $func->(@_);
            return _sanitizePerl($ret);
        };
    }
}

sub _sanitizePerl {
    my ($value, $seen) = @_;
    my $ref = ref $value;
    my $addr = refaddr $value;
    $seen ||= {};

    if (!defined $value) {
        return undef;
    } elsif ($ref) {
        if ($ref eq 'HASH') {
            return $seen->{$addr} ||= {map {$_ => _sanitizePerl($value->{$_}, $seen)} keys %$value};
        } elsif ($ref eq 'ARRAY') {
            return $seen->{$addr} ||= [map {_sanitizePerl($_, $seen)}  @$value];
        } elsif ($ref eq 'CODE') {
            return $seen->{$addr} ||= sub {
                my $ret = $value->(@_);
                return _sanitizePerl($ret);
            };
        } else {
            return undef;
        }
    } else {
        my $class = ref $CONTEXT->{adapter};

        no strict 'refs';
        my $encode_utf8   = ${$class.'::ENCODE_UTF8'};
        my $sanitize_utf8 = ${$class.'::SANITIZE_UTF8'};
        my $force_untaint = ${$class.'::FORCE_UNTAINT'};
        use strict 'refs';

        my $newRef = EJS::Template::Util::clean_text_ref(\$value,
                $encode_utf8, $sanitize_utf8, $force_untaint);

        return $$newRef;
    }
}

sub _perl2je {
    my ($value, $seen) = @_;
    my $engine = $CONTEXT->{engine};
    my $ref = ref $value;
    my $addr = refaddr $value;
    $seen ||= {};

    if (!defined $value) {
        return $engine->undefined();
    } elsif ($ref) {
        if ($ref eq 'HASH') {
            return $seen->{$addr} ||= $engine->upgrade({map {$_ => _perl2je($value->{$_}, $seen)} keys %$value});
        } elsif ($ref eq 'ARRAY') {
            return $seen->{$addr} ||= $engine->upgrade([map {_perl2je($_, $seen)}  @$value]);
        } elsif ($ref eq 'CODE') {
            return $seen->{$addr} ||= $engine->upgrade(sub {
                my $ret = $value->(map {_je2perl($_)} @_);
                return _perl2je($ret);
            });
        } else {
            # TODO: Throw an exception?
            return $engine->undefined();
        }
    } elsif (looks_like_number($value)) {
        return JE::Number->new($engine, $value);
    } else {
        return $engine->upgrade($value);
    }
}

sub _je2perl {
    my ($jeObj, $seen) = @_;
    my $ref = ref $jeObj;
    my $addr = refaddr $jeObj;
    $seen ||= {};

    if ($ref =~ /^JE::/) {
        if ($ref =~ /^JE::(?:Object::)?(?:Null|Undefined)$/) {
            return undef;
        } elsif ($ref =~ /^JE::(?:Object::)?(?:String|Number|Boolean|Date|RegExp)$/) {
            return $jeObj->value;
        } elsif ($ref =~ /^JE::Object::Function(?:$|::)/) {
            return $seen->{$addr} ||= sub {
                my $jeRet = $jeObj->(map {_perl2je($_)} @_);
                return _je2perl($jeRet);
            };
        } elsif ($ref =~ /^JE::Object::Array(?:$|::)/) {
            return $seen->{$addr} ||= [map {_je2perl($_, $seen)} @$jeObj];
        } elsif ($ref =~ /^JE::Object(?:$|::)/) {
            return $seen->{$addr} ||= {map {$_ => _je2perl($jeObj->{$_}, $seen)} keys %$jeObj};
        } else {
            return eval {$jeObj->value};
        }
    }

    return $jeObj;
}

sub _applyCallback {
    my ($callback, $values, $param) = @_;
    my $loop = {};
    my $results = [];

    for my $index (0..$#$values) {
        my $value = $values->[$index];

        $loop->{first} = ($index == 0 ? 1 : 0);
        $loop->{last}  = ($index == $#$values ? 1 : 0);
        $loop->{index} = $index;
        $loop->{value} = $value;

        my $result = $callback->($value, $loop);
        push @$results, $result if defined $result;
    }

    return $results;
}

sub _flattenParam {
    my ($args) = @_;

    my $param = {};
    my $positional = [];
    my $traverse;

    $traverse = sub {
        my ($args) = @_;

        for my $arg (@$args) {
            my $ref = ref $arg;

            if (!$ref) {
                push @$positional, $arg;
            } elsif ($ref eq 'HASH') {
                for my $key (keys %$arg) {
                    $param->{$key} = $arg->{$key};
                }
            } elsif ($ref eq 'ARRAY') {
                # Treat an array as a positional parameter
                push @$positional, $arg;
            } elsif ($ref eq 'CODE') {
                $param->{callback} = $arg;
            }
        }
    };

    $traverse->($args);
    return ($param, $positional);
}

sub _parseParam {
    my ($args, $fields) = @_;
    my ($param, $positional) = _flattenParam($args);

    # Scan specification ($fields)
    my $missingInParam = [];
    my $optionalCount = 0;
    my $isSpecified = {};
    my $isOptional = {};
    my $isWildcard = {};

    for my $rawField (@$fields) {
        my $field = $rawField;
        my $optional = ($field =~ s/\?$//);
        my $wildcard = ($field =~ s/\*$//);
        $optional = 1 if $wildcard;

        if (!exists $param->{$field}) {
            push @$missingInParam, [\$field, $optional];
            $optionalCount++ if $optional;
        }

        $isSpecified->{$field} = 1;
        $isOptional->{$field} = 1 if $optional;
        $isWildcard->{$field} = 1 if $wildcard;
    }

    # Assign scalar params
    my $min = @$missingInParam - $optionalCount;
    my $max = @$missingInParam;

    if (@$positional < $min) {
        (my $func = (caller 1)[3]) =~ s/.*://;
        die "Insufficient parameters: $func(".join(', ', @$fields).")\n";
    } elsif (@$positional > $max) {
        (my $func = (caller 1)[3]) =~ s/.*://;
        die "Too many parameters: $func(".join(', ', @$fields).")\n";
    }

    my $optionalToIgnore = @$missingInParam - @$positional;
    my $mappedFields = [];

    for my $pair (@$missingInParam) {
        if ($optionalToIgnore > 0 && $pair->[1]) {
            $optionalToIgnore--;
        } else {
            push @$mappedFields, $pair->[0];
        }
    }

    for my $i (0..$#$positional) {
        my $fieldRef = $mappedFields->[$i];
        $param->{$$fieldRef} = $positional->[$i];
    }

    # Expand variables
    for my $key (qw(web topic file toWeb toTopic toFile baseWeb user url)) {
        my $value = $param->{$key};

        if (defined $value && $value =~ /%/) {
            $TWiki::Plugins::SESSION->expandAllTags(\$param->{$key},
                $CONTEXT->{topic}, $CONTEXT->{web}, $CONTEXT->{meta});
        }
    }

    # Normalize Web.Topic
    for my $keys (['web', 'topic'], ['toWeb', 'toTopic']) {
        my ($webKey, $topicKey) = @$keys;

        if ($isSpecified->{$webKey} || $isSpecified->{$topicKey} ||
                defined $param->{$webKey} || defined $param->{$topicKey}) {
            my ($web, $topic) = ($param->{$webKey}, $param->{$topicKey});
            my ($origWeb, $origTopic) = ($web, $topic);
            my ($webEmpty, $topicEmpty);

            if (!defined $web || $web eq '') {
                $web = $CONTEXT->{web} || $TWiki::Plugins::SESSION->{webName};
                $webEmpty = 1;
            }

            if (!defined $topic || $topic eq '') {
                $topic = $CONTEXT->{topic} || $TWiki::Plugins::SESSION->{topicName};
                $topicEmpty = 1;
            } else {
                if ($topic =~ m{[./]}) {
                    $webEmpty = 0;
                }
            }

            ($web, $topic) = TWiki::Func::normalizeWebTopicName($web, $topic);
            
            if ($isWildcard->{$webKey} || $isWildcard->{$topicKey} ||
                    (!$isSpecified->{$webKey} && defined $param->{$webKey}) ||
                    (!$isSpecified->{$topicKey} && defined $param->{$topicKey})) {
                $topic = $origTopic if $topicEmpty;
                $web = $origWeb if $webEmpty;
            }

            ($param->{$webKey}, $param->{$topicKey}) = ($web, $topic);
        }
    }

    return $param;
}

sub _attrValue {
    my ($text) = @_;
    $text =~ s/"/\\"/g;
    return '"'.$text.'"';
}

sub _wildcard2regex {
    my ($wildcard) = @_;
    my $regex = quotemeta $wildcard;
    $regex =~ s/(?<!\\\\)\\\*/.*/g;
    return qr/^$regex$/;
}

sub _verifyChangePolicy {
    my ($action, $webs) = @_;
    my $session = $TWiki::Plugins::SESSION;
    my $query = $session->{request};

    # Check for POST policy
    if ($CONTEXT->{config}{postMethodPolicy}) {
        if ($query->request_method() ne 'POST') {
            die "POST method is required for action '$action'\n";
        }
    }

    # Check for Same-Web policy
    if ($CONTEXT->{config}{sameWebPolicy}) {
        my $baseWeb = $session->{SESSION_TAGS}{BASEWEB};
        my $quoted = quotemeta $baseWeb;
        my $regex = qr{^$quoted(?:$|[/\.])};

        for my $web (@{$webs || []}) {
            if ($web !~ $regex) {
                die "'$web' is outside the current web and cannot be modified\n";
            }
        }
    }

    # Check for CryptToken
    if ($TWiki::cfg{CryptToken}{Enable} && $CONTEXT->{config}{secureActions}{lc $action}) {
        eval {TWiki::UI::verifyCryptToken($session, $query->param('crypttoken'))};
        die "Invalid crypttoken\n" if $@;
    }
}

sub _getTopic {
    my ($web, $topic, $rev, $permType) = @_;

    # If the topic does not exist, TWiki::Func::readTopic() returns an empty meta
    my ($meta) = TWiki::Func::readTopic($web, $topic, $rev);

    my $allowed = TWiki::Func::checkAccessPermission($permType,
            TWiki::Func::getWikiName(), $meta->{_text}, $topic, $web, $meta);

    if (!$allowed) {
        die "$permType access denied: $web.$topic\n";
    }

    return $meta;
}

sub _parentWeb {
    my ($web) = @_;
    return defined $web && $web =~ s/[\/\.][^\/\.]+$// ? $web : undef;
}

sub _processInlineEJS {
    my ($textRef, $variables) = @_;
    $variables ||= {};

    # Workaround for EJS bug where Executor will unexpectedly overwrite print() and EJS object.
    my $isJE = $CONTEXT->{engine}->isa('JE');

    no strict 'refs';
    for my $name (@EJS_ROOT_FUNCTIONS) {
        $variables->{$name} = _wrapFunction(\&{'TWiki::Plugins::EmbeddedJSPlugin::Func::'.$name}, $isJE);
    }
    use strict 'refs';

    EJS::Template->context->process($textRef, $variables);
}

sub requireTopic {
    my $param = _parseParam(\@_, ['topic']);
    my $web = $param->{web};
    my $topic = $param->{topic};

    if (exists $CONTEXT->{requiredTopics}{"$web.$topic"}) {
        return 0;
    } else {
        $CONTEXT->{requiredTopics}{"$web.$topic"} = 1;
    }

    if (!TWiki::Func::topicExists($web, $topic)) {
        die "Topic '$web.$topic' does not exist\n";
    }

    my $meta = _getTopic($web, $topic, $param->{rev}, 'VIEW');

    local $CONTEXT->{web}   = $web;
    local $CONTEXT->{topic} = $topic;
    local $CONTEXT->{meta}  = $meta;

    _processInlineEJS(\$meta->{_text});

    return 1;
}

sub _print {
    if ($CAPTURE) {
        push @$CAPTURE, map {ref($_) ? JSON_stringify($_) : $_} @_;
    } else {
        EJS::Template->context->print(map {ref($_) ? JSON_stringify($_) : $_} @_);
    }
}

sub print {
    _print(@_);
    return undef;
}

sub println {
    _print(@_, "\n");
    return undef;
}

sub findWebs {
    my $param = _parseParam(\@_, ['web*']);
    my $web = $param->{web};

    my $filter = $param->{filter};
    my @webs;

    if (!defined $web || $web eq '' || $web eq '*') {
        @webs = $TWiki::Plugins::SESSION->{store}->getListOfWebs($filter, undef, 1);
    } else {
        for my $chunk (split m{/}, $web) {
            my @subwebs;

            if ($chunk =~ /\*/) {
                my $regex = _wildcard2regex($chunk);

                if (@webs) {
                    @subwebs = map {$TWiki::Plugins::SESSION->{store}->getListOfWebs($filter, $_, 1)} @webs;
                } else {
                    @subwebs = $TWiki::Plugins::SESSION->{store}->getListOfWebs($filter, undef, 1);
                }

                @subwebs = grep {basename($_) =~ $regex} @subwebs;
            } else {
                if (@webs) {
                    @subwebs = grep {TWiki::Func::webExists($_)} map {"$_/$chunk"} @webs;
                } else {
                    @subwebs = ($chunk) if TWiki::Func::webExists($chunk);
                }
            }

            @webs = @subwebs;
            last unless @webs;
        }
    }
    
    if (my $callback = $param->{callback}) {
        return _applyCallback($callback, \@webs, $param);
    } else {
        return \@webs;
    }
}

sub findTopics {
    my $param = _parseParam(\@_, ['topic*']);
    my $topic = $param->{topic};

    my @webs;
    my $webSpecified = defined $param->{web};

    if ($webSpecified) {
        if ($param->{web} =~ /\*/) {
            @webs = @{findWebs($param->{web}, {filter => $param->{filter}})};
        } elsif (TWiki::Func::webExists($param->{web})) {
            @webs = ($param->{web});
        }
    } else {
        @webs = ($TWiki::Plugins::SESSION->{webName});
    }

    my @topics;

    for my $web (@webs) {
        my @foundTopics;

        if (!defined $topic || $topic eq '' || $topic eq '*') {
            @foundTopics = TWiki::Func::getTopicList($web);
        } elsif ($topic =~ /\*/) {
            my $regex = _wildcard2regex($topic);
            my @found = TWiki::Func::getTopicList($web);
            @foundTopics = grep {$_ =~ $regex} @found;
        } else {
            if (TWiki::Func::topicExists($web, $topic)) {
                @foundTopics = ($topic);
            }
        }

        if ($webSpecified) {
            @foundTopics = map {"$web.$_"} @foundTopics;
        }

        push @topics, @foundTopics;
    }

    if (my $callback = $param->{callback}) {
        my $ret = _applyCallback($callback, \@topics, $param);
        return $ret;
    } else {
        return \@topics;
    }
}

sub webExists {
    my $param = _parseParam(\@_, ['web']);
    my $web = $param->{web};
    return TWiki::Func::webExists($web) ? 1 : 0;
}

sub topicExists {
    my $param = _parseParam(\@_, ['topic']);
    my $web = $param->{web};
    my $topic = $param->{topic};
    return TWiki::Func::topicExists($web, $topic) ? 1 : 0;
}

sub createWeb {
    my $param = _parseParam(\@_, ['web']);
    _verifyChangePolicy('createweb', [$param->{web}]);

    my $web = $param->{web};
    my $baseWeb = $param->{baseWeb};

    unless (defined $baseWeb) {
        $baseWeb = $CONTEXT->{config}{defaultBaseWeb};
    }

    my $session = $TWiki::Plugins::SESSION;
    my $cUID = $session->{user};
    my $parentWeb = _parentWeb($web);

    # Check metadata
    if ($TWiki::cfg{Mdrepo}{WebRecordRequired} && $session->{mdrepo} &&
            !$session->{mdrepo}->getRec('webs', TWiki::topLevelWeb($web))) {
        die "No metadata exists for web '$web'\n";
    }

    # Check existence
    if (TWiki::Func::webExists($web)) {
        die "Web '$web' already exists\n";
    } elsif (TWiki::Func::topicExists(undef, $web)) {
        die "Topic '$web' already exists\n";
    } elsif ($parentWeb && !TWiki::Func::webExists($parentWeb)) {
        die "Web '$parentWeb' does not exist\n";
    } elsif (!TWiki::Func::webExists($baseWeb)) {
        die "Base web '$baseWeb' does not exist\n";
    }

    # Check permission
    if ($parentWeb) {
        if (!$session->{users}->canCreateWeb($parentWeb)) {
            if (!TWiki::Func::checkAccessPermission('CHANGE',
                    TWiki::Func::getWikiName(), undef, undef, $parentWeb, undef)) {
                die "CREATEWEB access denied: $parentWeb\n";
            }
        }
    }

    # Execute changes
    $session->{store}->createWeb($cUID, $web, $baseWeb, undef);

    return 1;
}

sub moveWeb {
    my $param = _parseParam(\@_, ['web', 'toWeb']);
    _verifyChangePolicy('renameweb', [$param->{web}, $param->{toWeb}]);

    my $fromWeb = $param->{web};
    my $toWeb = $param->{toWeb};

    my $session = $TWiki::Plugins::SESSION;
    my $cUID = $session->{user};
    my $parentWeb = _parentWeb($toWeb);

    # Check metadata
    if ($TWiki::cfg{Mdrepo}{WebRecordRequired} && $session->{mdrepo} &&
            !$session->{mdrepo}->getRec('webs', TWiki::topLevelWeb($toWeb))) {
        die "No metadata exists for web '$toWeb'\n";
    }

    # Check existence
    if (!TWiki::Func::webExists($fromWeb)) {
        die "Web '$fromWeb' does not exist\n";
    }

    if (TWiki::Func::webExists($toWeb)) {
        die "Web '$toWeb' already exists\n";
    } elsif (TWiki::Func::topicExists(undef, $toWeb)) {
        die "Topic '$toWeb' already exists\n";
    } elsif ($parentWeb && !TWiki::Func::webExists($parentWeb)) {
        die "Web '$parentWeb' does not exist\n";
    }

    # Check permission
    if (!$session->{users}->canRenameWeb($fromWeb, $toWeb)) {
        if (!TWiki::Func::checkAccessPermission('CHANGE',
                TWiki::Func::getWikiName(), undef, undef, $fromWeb, undef)) {
            die "RENAMEWEB access denied: $fromWeb, $toWeb\n";
        }

        if ($parentWeb) {
            if (!TWiki::Func::checkAccessPermission('CHANGE',
                    TWiki::Func::getWikiName(), undef, undef, $parentWeb, undef)) {
                die "CHANGE access denied: $parentWeb\n";
            }
        }
    }

    # TODO: Run something similar to TWiki::UI::Manage::_updateWebReferringTopics?

    # Execute changes
    $session->{store}->moveWeb($fromWeb, $toWeb, $cUID);

    return 1;
}

sub removeWeb {
    my $param = _parseParam(\@_, ['web']);
    _verifyChangePolicy('renameweb', [$param->{web}]);

    my $fromWeb = $param->{web};
    my $toWeb = $param->{toWeb};

    my $session = $TWiki::Plugins::SESSION;
    my $cUID = $session->{user};
    my $trashWeb = $session->trashWebName(web => $fromWeb);
    my $parentWeb = _parentWeb($toWeb);

    if (defined $toWeb) {
        if (substr($toWeb, 0, length $trashWeb + 1) ne "$trashWeb/") {
            die "Web '$toWeb' must be under '$trashWeb'\n";
        }
    } else {
        (my $name = $fromWeb) =~ s/[\/\.]//g;
        $toWeb .= "$trashWeb/$name";

        my $base = $toWeb;
        my $next = 1;

        while (TWiki::Func::webExists($toWeb) || TWiki::Func::topicExists(undef, $toWeb)) {
            $toWeb = $base.$next;
            $next++;
        }
    }

    # Check existence
    if (!TWiki::Func::webExists($fromWeb)) {
        die "Web '$fromWeb' does not exist\n";
    }

    if (TWiki::Func::webExists($toWeb)) {
        die "Web '$toWeb' already exists\n";
    } elsif (TWiki::Func::topicExists(undef, $toWeb)) {
        die "Topic '$toWeb' already exists\n";
    } elsif ($parentWeb && !TWiki::Func::webExists($parentWeb)) {
        die "Web '$parentWeb' does not exist\n";
    }

    # Check permission
    if (!$session->{users}->canRenameWeb($fromWeb, $toWeb)) {
        if (!TWiki::Func::checkAccessPermission('CHANGE',
                TWiki::Func::getWikiName(), undef, undef, $fromWeb, undef)) {
            die "RENAMEWEB access denied: $fromWeb, $toWeb\n";
        }

        if ($parentWeb) {
            if (!TWiki::Func::checkAccessPermission('CHANGE',
                    TWiki::Func::getWikiName(), undef, undef, $parentWeb, undef)) {
                die "CHANGE access denied: $parentWeb\n";
            }
        }
    }

    # Execute changes
    $session->{store}->moveWeb($fromWeb, $toWeb, $cUID);

    return 1;
}

sub readTopic {
    my $param = _parseParam(\@_, ['topic']);
    my $web = $param->{web};
    my $topic = $param->{topic};

    if (!TWiki::Func::topicExists($web, $topic)) {
        die "Topic '$web.$topic' does not exist\n";
    }

    my $meta = _getTopic($web, $topic, $param->{rev}, 'VIEW');
    return $meta->{_text};
}

sub _extractFormFields {
    my ($web, $topic, $meta, $param) = @_;
    my $formName;
    my $isAdded;

    if (exists $param->{formName}) {
        $formName = $param->{formName};

        if (defined $formName && $formName ne '') {
            $meta->put('FORM', {name => $formName});
            $isAdded = 1;
        } else {
            $meta->remove('FORM');
        }
    } else {
        my $form = $meta->get('FORM');
        $formName = $form->{name} if $form;
    }

    if (defined $formName && $formName ne '') {
        require TWiki::Form;
        my $formDef = TWiki::Form->new($TWiki::Plugins::SESSION, $web, $formName);
        die "No such form template: $formName\n" unless $formDef;

        for my $fieldDef (@{$formDef->getFields()}) {
            next unless $fieldDef->{name};
            my $value = $param->{$fieldDef->{name}};

            if (defined $value) {
                $value =~ s/^\s+|\s+$//g;
                $value = '1' if $fieldDef->isMandatory() && $value eq '';

                $meta->putKeyed('FIELD', {
                    name  => $fieldDef->{name},
                    title => $fieldDef->{title},
                    value => $value,
                });
            } elsif ($isAdded) {
                $value = $fieldDef->isMandatory() ? '1' : '';

                $meta->putKeyed('FIELD', {
                    name  => $fieldDef->{name},
                    title => $fieldDef->{title},
                    value => $value,
                });
            }
        }
    }

    if (exists $param->{parentTopic}) {
        my $parentTopic = $param->{parentTopic};

        if (!defined $parentTopic || $parentTopic eq '' || $parentTopic eq 'none') {
            $meta->remove('TOPICPARENT');
        } else {
            $meta->put('TOPICPARENT', {name => $param->{parentTopic}});
        }
    }
}

sub _saveTopic {
    my ($web, $topic, $param) = @_;

    # Check existence
    if (!TWiki::Func::webExists($web)) {
        die "Web '$web' does not exist\n";
    }

    # Check permission
    my $meta = _getTopic($web, $topic, undef, 'CHANGE');

    # Check lease conflict
    if (!$param->{breakLock}) {
        _getEditLockNoConflict($web, $topic);
    }

    $meta->{_text} = $param->{text} if defined $param->{text};
    _extractFormFields($web, $topic, $meta, $param);

    my $options = {
        forcenewrevision => TWiki::isTrue($param->{forceNewRevision}),
        minor => TWiki::isTrue($param->{minor}),
    };

    my $store = $TWiki::Plugins::SESSION->{store};
    my $user = $TWiki::Plugins::SESSION->{user};
    $store->saveTopic($user, $web, $topic, $meta->{_text}, $meta, $options);
    $store->clearLease($web, $topic);

    return 1;
}

sub saveTopic {
    my $param = _parseParam(\@_, ['topic', 'text']);
    _verifyChangePolicy('save', [$param->{web}]);

    my $web = $param->{web};
    my $topic = $param->{topic};

    return _saveTopic($web, $topic, $param);
}

sub moveTopic {
    my $param = _parseParam(\@_, ['topic', 'toTopic']);
    _verifyChangePolicy('rename', [$param->{web}, $param->{toWeb}]);

    my $fromWeb   = $param->{web};
    my $fromTopic = $param->{topic};
    my $toWeb     = $param->{toWeb};
    my $toTopic   = $param->{toTopic};

    # Check existence
    if (!TWiki::Func::topicExists($fromWeb, $fromTopic)) {
        die "Topic '$fromWeb.$fromTopic' does not exist\n";
    } elsif (!TWiki::Func::webExists($toWeb)) {
        die "Web '$toWeb' does not exist\n";
    } elsif (TWiki::Func::topicExists($toWeb, $toTopic)) {
        die "Topic '$toWeb.$toTopic' already exists\n";
    }

    # Check permission
    _getTopic($fromWeb, $fromTopic, undef, 'CHANGE');
    _getTopic($toWeb, $toTopic, undef, 'CHANGE');

    # Check lease conflict
    if (!$param->{breakLock}) {
        _getEditLockNoConflict($fromWeb, $fromTopic);
    }

    # Execute changes
    TWiki::Func::moveTopic($fromWeb, $fromTopic, $toWeb, $toTopic);

    return 1;
}

sub removeTopic {
    my $param = _parseParam(\@_, ['topic']);
    _verifyChangePolicy('rename', [$param->{web}]);

    my $fromWeb   = $param->{web};
    my $fromTopic = $param->{topic};
    my $toWeb     = $param->{toWeb};
    my $toTopic   = $param->{toTopic};
    my $trashWeb  = $TWiki::Plugins::SESSION->trashWebName(web => $fromWeb);

    if (defined $toWeb) {
        if ($toWeb ne $trashWeb && substr($toWeb, 0, length($trashWeb) + 1) ne "$trashWeb/") {
            die "Web '$toWeb' must be under '$trashWeb'\n";
        }
    } else {
        $toWeb = $trashWeb;
    }

    if (!defined $toTopic) {
        $toTopic = $fromWeb.$fromTopic;
        $toTopic =~ s{[/\.]+}{}g;

        # Imitate algorithm in TWiki::UI::Manage::_newTopicScreen
        my $base = $toTopic;
        my $next = 1;

        while (TWiki::Func::topicExists($toWeb, $toTopic)) {
            $toTopic = $base.$next;
            $next++;
        }
    }

    # Check existence
    if (!TWiki::Func::topicExists($fromWeb, $fromTopic)) {
        die "Topic '$fromWeb.$fromTopic' does not exist\n";
    } elsif (TWiki::Func::topicExists($toWeb, $toTopic)) {
        die "Topic '$toWeb.$toTopic' already exists\n";
    }

    # Check permission
    _getTopic($fromWeb, $fromTopic, undef, 'CHANGE');

    # Check lease conflict
    if (!$param->{breakLock}) {
        _getEditLockNoConflict($fromWeb, $fromTopic);
    }

    # Execute changes
    TWiki::Func::moveTopic($fromWeb, $fromTopic, $toWeb, $toTopic);

    return 1;
}

sub getFormName {
    my $param = _parseParam(\@_, ['topic?']);
    my $web = $param->{web};
    my $topic = $param->{topic};

    if (!TWiki::Func::topicExists($web, $topic)) {
        die "Topic '$web.$topic' does not exist\n";
    }

    my $meta = _getTopic($web, $topic, $param->{rev}, 'VIEW');

    if (my $form = $meta->get('FORM')) {
        return $form->{name};
    } else {
        return undef;
    }
}

sub setFormName {
    my $param = _parseParam(\@_, ['topic?', 'name']);
    _verifyChangePolicy('save', [$param->{web}]);

    my $web = $param->{web};
    my $topic = $param->{topic};

    if (!TWiki::Func::topicExists($web, $topic)) {
        die "Topic '$web.$topic' does not exist\n";
    }

    return _saveTopic($web, $topic, {
        formName => $param->{name}
    });
}

sub getFormField {
    my $param = _parseParam(\@_, ['topic?', 'name']);
    my $web = $param->{web};
    my $topic = $param->{topic};

    if (!TWiki::Func::topicExists($web, $topic)) {
        die "Topic '$web.$topic' does not exist\n";
    }

    my $meta = _getTopic($web, $topic, $param->{rev}, 'VIEW');

    if (my $field = $meta->get('FIELD', $param->{name})) {
        return $field->{value};
    } else {
        return undef;
    }
}

sub setFormField {
    my $param = _parseParam(\@_, ['topic?', 'name', 'value']);
    _verifyChangePolicy('save', [$param->{web}]);

    my $web = $param->{web};
    my $topic = $param->{topic};

    if (!TWiki::Func::topicExists($web, $topic)) {
        die "Topic '$web.$topic' does not exist\n";
    }

    return _saveTopic($web, $topic, {
        $param->{name} => $param->{value}
    });
}

sub getFormFields {
    my $param = _parseParam(\@_, ['topic?']);
    my $web = $param->{web};
    my $topic = $param->{topic};

    if (!TWiki::Func::topicExists($web, $topic)) {
        die "Topic '$web.$topic' does not exist\n";
    }

    my $meta = _getTopic($web, $topic, $param->{rev}, 'VIEW');
    my $fields = {};

    for my $field ($meta->find('FIELD')) {
        $fields->{$field->{name}} = $field->{value};
    }

    return $fields;
}

sub checkPermission {
    my $param = _parseParam(\@_, ['topic?', 'type']);
    my $web = $param->{web};
    my $topic = $param->{topic};

    if (!TWiki::Func::topicExists($web, $topic)) {
        die "Topic '$web.$topic' does not exist\n";
    }

    my $ret = eval {
        my $meta = _getTopic($web, $topic, $param->{rev}, 'VIEW');
        my $user = $param->{user};
        $user = TWiki::Func::getWikiName() if !defined $user || $user eq '';
        TWiki::Func::checkAccessPermission($param->{type}, $user, $meta->{_text}, $topic, $web, $meta);
    };

    if ($@ && $@ !~ /^[A-Z]+ access denied/) {
        die $@;
    }

    return $ret ? 1 : 0;
}

sub getRevisionInfo {
    my $param = _parseParam(\@_, ['topic?']);
    my $web = $param->{web};
    my $topic = $param->{topic};

    # Check existence
    if (!TWiki::Func::topicExists($web, $topic)) {
        die "Topic '$web.$topic' does not exist\n";
    }

    # Check permission
    _getTopic($web, $topic, $param->{rev}, 'VIEW');

    my ($time, $user, $rev, $comment) = TWiki::Func::getRevisionInfo($web, $topic, $param->{rev});
    my $userName = TWiki::Func::wikiToUserName(TWiki::Func::getWikiName($user));

    my $info = {
        time     => $time,
        user     => $user,
        userName => $userName,
        rev      => $rev,
        comment  => $comment,
    };

    return $info;
}

sub getRevisionAtTime {
    my $param = _parseParam(\@_, ['topic?', 'time']);
    my $web   = $param->{web};
    my $topic = $param->{topic};
    my $time  = $param->{time};

    # Check existence
    if (!TWiki::Func::topicExists($web, $topic)) {
        die "Topic '$web.$topic' does not exist\n";
    }

    # Check permission
    _getTopic($web, $topic, undef, 'VIEW');

    return TWiki::Func::getRevisionAtTime($web, $topic, $time);
}

sub _getEditLock {
    my ($web, $topic) = @_;

    my $session = $TWiki::Plugins::SESSION;
    my $users = $session->{users};

    my $lease = $session->{store}->getLease($web, $topic);

    if ($lease) {
        my $conflict = '';
        my $now = time();

        if ($now <= $lease->{expires}) {
            $conflict = 'lease_active';
        } elsif ($TWiki::cfg{LeaseLengthLessForceful} < 0 ||
                $now < $lease->{expires} + $TWiki::cfg{LeaseLengthLessForceful}) {
            $conflict = 'lease_old';
        } else {
            # Lease has expired
            return undef;
        }

        my $leaseUser = $users->webDotWikiName($lease->{user});
        my $currentUser = $users->webDotWikiName($session->{user});

        if ($leaseUser eq $currentUser) {
            $conflict = '';
        }

        my $userName = TWiki::Func::wikiToUserName(TWiki::Func::getWikiName($lease->{user}));

        return {
            user     => $lease->{user},
            expires  => int $lease->{expires},
            taken    => int $lease->{taken},
            userName => $userName, # comparable to getUserName() return value
            conflict => $conflict,
        };
    } else {
        return undef;
    }
}

sub _getEditLockNoConflict {
    my ($web, $topic) = @_;
    my $editLock = _getEditLock($web, $topic);

    if ($editLock) {
        if (my $conflict = $editLock->{conflict}) {
            my $users = $TWiki::Plugins::SESSION->{users};

            if ($conflict eq 'lease_active') {
                my $wikiName = $users->getWikiName($editLock->{user});
                die "Lease conflict: $wikiName is editing [[$web.$topic][$topic]]\n";
            } elsif ($conflict eq 'lease_old') {
                my $wikiName = $users->getWikiName($editLock->{user});
                die "Lease conflict: $wikiName may still be editing [[$web.$topic][$topic]]\n";
            }
        }
    }

    return $editLock;
}

sub getEditLock {
    my $param = _parseParam(\@_, ['topic']);
    my $web   = $param->{web};
    my $topic = $param->{topic};

    # Check existence
    if (!TWiki::Func::topicExists($web, $topic)) {
        die "Topic '$web.$topic' does not exist\n";
    }

    # Check permission
    _getTopic($web, $topic, undef, 'VIEW');

    return _getEditLock($web, $topic);
}

sub acquireEditLock {
    my $param = _parseParam(\@_, ['topic']);
    my $web   = $param->{web};
    my $topic = $param->{topic};

    # Check existence
    if (!TWiki::Func::topicExists($web, $topic)) {
        die "Topic '$web.$topic' does not exist\n";
    }

    # Check permission
    _getTopic($web, $topic, undef, 'CHANGE');

    my $editLock = _getEditLockNoConflict($web, $topic);

    my $length = $TWiki::cfg{LeaseLength}; # TODO: parameterize?
    my $leaseDelay = 1 * 60; # TODO: make it configurable?

    if ($editLock) {
        # If the existing lease is fresh enough, don't attempt to create a new lease
        my $now = time();

        if ($editLock->{expires} >= $now + $length - $leaseDelay &&
                $editLock->{taken} >= $now - $leaseDelay) {
            return 0;
        }
    }

    my $session = $TWiki::Plugins::SESSION;
    my $user = $session->{user};
    $session->{store}->setLease($web, $topic, $user, $length);

    return 1;
}

sub releaseEditLock {
    my $param = _parseParam(\@_, ['topic']);
    my $web   = $param->{web};
    my $topic = $param->{topic};

    # Check existence
    if (!TWiki::Func::topicExists($web, $topic)) {
        die "Topic '$web.$topic' does not exist\n";
    }

    my $editLock = _getEditLock($web, $topic);

    if ($editLock && !$editLock->{conflict}) {
        $TWiki::Plugins::SESSION->{store}->clearLease($web, $topic);
        return 1;
    } else {
        return 0;
    }
}

sub getAttachmentList {
    my $param = _parseParam(\@_, ['topic?']);
    my $web   = $param->{web};
    my $topic = $param->{topic};

    # Check existence
    if (!TWiki::Func::topicExists($web, $topic)) {
        die "Topic '$web.$topic' does not exist\n";
    }

    # Check permission
    my $meta = _getTopic($web, $topic, undef, 'VIEW');

    return [map {$_->{name}} $meta->find('FILEATTACHMENT')];
}

sub attachmentExists {
    my $param = _parseParam(\@_, ['topic?', 'file']);
    my $web   = $param->{web};
    my $topic = $param->{topic};
    my $file  = $param->{file};
    return TWiki::Func::attachmentExists($web, $topic, $file) ? 1 : 0;
}

sub readAttachment {
    my $param = _parseParam(\@_, ['topic?', 'file']);
    my $web   = $param->{web};
    my $topic = $param->{topic};
    my $file  = $param->{file};
    my $rev   = $param->{rev};

    # Check existence
    if (!TWiki::Func::attachmentExists($web, $topic, $file)) {
        die "Attachment '$web.$topic/$file' does not exist\n";
    }

    # Check permission
    _getTopic($web, $topic, undef, 'VIEW');

    return TWiki::Func::readAttachment($web, $topic, $file, $rev);
}

sub saveAttachment {
    my $param = _parseParam(\@_, ['topic?', 'file', 'data']);
    _verifyChangePolicy('upload', [$param->{web}]);

    my $web     = $param->{web};
    my $topic   = $param->{topic};
    my $file    = $param->{file};
    my $dataRef = \$param->{data} if defined $param->{data};

    # Check existence
    if (!TWiki::Func::topicExists($web, $topic)) {
        die "Topic '$web.$topic' does not exist\n";
    }

    # Check permission
    my $meta = _getTopic($web, $topic, undef, 'CHANGE');

    my $options = {
        comment  => $param->{comment},
        hide     => $param->{hide},
        filepath => $param->{filepath},
        filedate => $param->{filedate},
    };

    my $stream  = IO::String->new($dataRef);

    TWiki::Func::saveAttachment($web, $topic, $file, {
        comment     => $options->{comment},
        hide        => $options->{hide},
        stream      => $stream,
        file        => $file,
        filepath    => $options->{filepath},
        filesize    => length($$dataRef),
        filedate    => $options->{filedate} || time(),
        tmpFilename => $file,
    });

    return 1;
}

sub moveAttachment {
    my $param = _parseParam(\@_, ['topic?', 'file', 'toTopic?', 'toFile']);
    _verifyChangePolicy('rename', [$param->{web}, $param->{toWeb}]);

    my $fromWeb   = $param->{web};
    my $fromTopic = $param->{topic};
    my $fromFile  = $param->{file};
    my $toWeb     = $param->{toWeb};
    my $toTopic   = $param->{toTopic};
    my $toFile    = $param->{toFile};

    # Check existence
    if (!TWiki::Func::attachmentExists($fromWeb, $fromTopic, $fromFile)) {
        die "Attachment '$fromWeb.$fromTopic/$fromFile' does not exist\n";
    } elsif (!TWiki::Func::topicExists($toWeb, $toTopic)) {
        die "Topic '$toWeb.$toTopic' does not exist\n";
    } elsif (TWiki::Func::attachmentExists($toWeb, $toTopic, $toFile)) {
        die "Attachment '$toWeb.$toTopic/$toFile' already exists\n";
    }

    # Check permission
    _getTopic($fromWeb, $fromTopic, undef, 'CHANGE');
    _getTopic($toWeb, $toTopic, undef, 'CHANGE');

    TWiki::Func::moveAttachment($fromWeb, $fromTopic, $fromFile, $toWeb, $toTopic, $toFile);
    return 1;
}

sub removeAttachment {
    my $param = _parseParam(\@_, ['topic?', 'file']);
    _verifyChangePolicy('rename', [$param->{web}]);

    my $fromWeb   = $param->{web};
    my $fromTopic = $param->{topic};
    my $fromFile  = $param->{file};
    my $toWeb     = $param->{toWeb};
    my $toTopic   = $param->{toTopic};
    my $toFile    = $param->{toFile};
    my $trashWeb  = $TWiki::Plugins::SESSION->trashWebName(web => $fromWeb);

    if (defined $toWeb) {
        if ($toWeb ne $trashWeb && substr($toWeb, 0, length($trashWeb) + 1) ne "$trashWeb/") {
            die "Web '$toWeb' must be under '$trashWeb'\n";
        }
    } else {
        $toWeb = $trashWeb;
    }

    if (!defined $toTopic) {
        $toTopic = 'TrashAttachment';
    }

    if (!defined $toFile) {
        $toFile = $fromFile;

        # Imitate algorithm in TWiki::UI::Manage::move
        if (TWiki::Func::attachmentExists($toWeb, $toTopic, $toFile)) {
            my $base = $toFile;
            my $ext = ($base =~ s/(\.[^\.]+)$//) ? $1 : '';
            my $next = 2;

            do {
                $toFile = $base.$next.$ext;
                $next++;
            } while (TWiki::Func::attachmentExists($toWeb, $toTopic, $toFile));
        }
    }

    # Check existence
    if (!TWiki::Func::attachmentExists($fromWeb, $fromTopic, $fromFile)) {
        die "Attachment '$fromWeb.$fromTopic/$fromFile' does not exist\n";
    } elsif (!TWiki::Func::topicExists($toWeb, $toTopic)) {
        die "Topic '$toWeb.$toTopic' does not exist\n";
    } elsif (TWiki::Func::attachmentExists($toWeb, $toTopic, $toFile)) {
        die "Attachment '$toWeb.$toTopic/$toFile' already exists\n";
    }

    # Check permission
    _getTopic($fromWeb, $fromTopic, undef, 'CHANGE');
    _getTopic($toWeb, $toTopic, undef, 'CHANGE');

    TWiki::Func::moveAttachment($fromWeb, $fromTopic, $fromFile, $toWeb, $toTopic, $toFile);
    return 1;
}

sub getMethod {
    my $query = $TWiki::Plugins::SESSION->{request};
    return $query->request_method();
}

sub isPost {
    my $query = $TWiki::Plugins::SESSION->{request};
    my $method = $query->request_method();
    return $method =~ /^POST$/i ? 1 : 0;
}

sub getParam {
    my $param = _parseParam(\@_, ['name', 'default?']);
    my $query = $TWiki::Plugins::SESSION->{request};
    my $value = $query->param($param->{name});
    $value = $param->{default} unless defined $value;
    return $value;
}

sub setParam {
    my $param = _parseParam(\@_, ['name', 'value']);
    my $query = $TWiki::Plugins::SESSION->{request};
    $query->param($param->{name}, $param->{value});
    return 1;
}

sub getUserName {
    my $param = _parseParam(\@_, ['user?']);
    return TWiki::Func::wikiToUserName(TWiki::Func::getWikiName($param->{user}));
}

sub getWikiName {
    my $param = _parseParam(\@_, ['user?']);
    return TWiki::Func::getWikiName($param->{user});
}

sub getWikiUserName {
    my $param = _parseParam(\@_, ['user?']);
    return TWiki::Func::getWikiUserName($param->{user});
}

sub getVariable {
    my $name = shift;
    my $default;
    my $param = {};

    for my $arg (@_) {
        my $ref = ref $arg;

        if ($ref) {
            if ($ref eq 'HASH') {
                $param = \%$arg;
            }
        } else {
            $default = $arg;
        }
    }

    my $chunks = [];
    push @$chunks, _attrValue($default) if defined $default;
    push @$chunks, $_, '=', _attrValue($param->{$_}) for keys %$param;
    my $attrs = join(' ', @$chunks);
    $attrs = undef if $attrs eq '';

    my $text = '%'.$name.(!defined $attrs ? '' : "{$attrs}").'%';

    my $topic = $CONTEXT->{topic};
    my $web   = $CONTEXT->{web};
    my $meta  = $CONTEXT->{meta};

    $TWiki::Plugins::SESSION->expandAllTags(\$text, $topic, $web, $meta);
    return $text;
}

sub setVariable {
    my ($name, $value) = @_;
    $TWiki::Plugins::SESSION->{prefs}->setPreferencesValue($name, $value);
    return 1;
}

sub expandVariables {
    my $text = shift;

    my $topic = $CONTEXT->{topic};
    my $web   = $CONTEXT->{web};
    my $meta  = $CONTEXT->{meta};

    $TWiki::Plugins::SESSION->expandAllTags(\$text, $topic, $web, $meta);
    return $text;
}

sub expandAutoInc {
    my $param = _parseParam(\@_, ['topic*']);
    my $web   = $param->{web};
    my $topic = $param->{topic};

    my $webSpecified = defined $web;
    my $topicSpecified = defined $topic;

    if (!$webSpecified) {
        $web = $CONTEXT->{web};
    }

    if ($topic =~ /^(.*?)AUTOINC(\d+)(.*)$/) {
        my ($prefix, $digits, $suffix) = ($1, $2, $3);
        my $pattern = '^'.quotemeta($prefix).'(\d+)'.quotemeta($suffix).'$';

        # SubWebs + Topics
        my @names = $TWiki::Plugins::SESSION->{store}->getListOfWebs(undef, $web, 1);
        push @names, TWiki::Func::getTopicList($web);

        my $max = $digits;
        $max =~ s/^0*(\d+)/$1/;

        for my $name (@names) {
            if ($name =~ /$pattern/) {
                my $num = $1;
                $num =~ s/^0*(\d+)/$1/;
                $num++;
                $max = $num if $max < $num;
            }
        }

        $digits = sprintf('%0'.length($digits).'d', $max);
        $topic = $prefix.$digits.$suffix;
    }

    return ($webSpecified ? "$web.$topic" : $topic);
}

sub formatText {
    my ($text, $vars) = @_;
    $vars ||= {};

    $text =~ s!(\$(\w+)|\{(.*?)\})!
        my $name = defined $2 ? $2 : $3;
        defined $vars->{$name} ? $vars->{$name} : $1!eg;

    $text = TWiki::expandStandardEscapes($text);

    return $text;
}

sub isTrue {
    my ($value, $default) = @_;
    return TWiki::isTrue($value, $default) ? 1 : 0;
}

sub normalizeWebTopicName {
    my $param = _parseParam(\@_, ['web?', 'topic']);
    return {web => $param->{web}, topic => $param->{topic}};
}

sub escapeHTML {
    return encode_entities($_[0]);
}

sub unescapeHTML {
    return decode_entities($_[0]);
}

sub _makeRegex {
    my @chars;
    my @strs;

    for my $pat (@_) {
        if (length $pat == 1) {
            push @chars, $pat;
        } else {
            push @strs, quotemeta $pat;
        }
    }

    if (@chars) {
        push @strs, '['.quotemeta(join('', @chars)).']';
    }

    return '('.join('|', @strs).')';
}

my $escapeTable = {
    "|"  => '%VBAR%',
    "\x00" => "\\0",
    "\x08" => "\\b",
    "\x09" => "\\t",
    "\x0A" => "\\n",
    "\x0B" => "\\v",
    "\x0C" => "\\f",
    "\x0D" => "\\r",
    "\\" => "\\\\",
};

my $unescapeTable = {map {$escapeTable->{$_} => $_} keys %$escapeTable};

my $escapeTableRE = _makeRegex(keys %$escapeTable);
my $unescapeTableRE = _makeRegex(keys %$unescapeTable);

sub escapeTable {
    my ($text) = @_;
    $text =~ s/$escapeTableRE/$escapeTable->{$1}/go;
    $text = escapeHTML($text);
    return $text;
}

sub unescapeTable {
    my ($text) = @_;
    $text = unescapeHTML($text);
    $text =~ s/$unescapeTableRE/$unescapeTable->{$1}/go;
    return $text;
}

my $escapeQuote = {
    "\0"   => "\\0",
    "\x08" => "\\b",
    "\x09" => "\\t",
    "\x0A" => "\\n",
    "\x0B" => "\\v",
    "\x0C" => "\\f",
    "\x0D" => "\\r",
    '"'    => "\\\"",
    "'"    => "\\'",
    "\\"   => "\\\\",
};

my $unescapeQuote = {map {$escapeQuote->{$_} => $_} keys %$escapeQuote};

my $escapeQuoteRE = _makeRegex(keys %$escapeQuote);
my $unescapeQuoteRE = _makeRegex(keys %$unescapeQuote);

sub escapeQuote {
    my $value = shift;
    $value =~ s/$escapeQuoteRE/$escapeQuote->{$1}/go;
    return $value;
}

sub unescapeQuote {
    my $value = shift;
    $value =~ s/$unescapeQuoteRE/$unescapeQuote->{$1}/go;
    return $value;
}

sub _parseFields {
    my ($fields) = @_;

    if ($fields) {
        if (!ref $fields) {
            $fields = [map {s/^\s+|\s+$//g; $_} split(/\s*,\s*/, $fields)];
        }
    }

    return $fields;
}

sub _addToTable {
    my ($table, $row) = @_;

    if (!$table->{headerAdded} && $table->{header}) {
        $table->{fields} ||= $row;
        $table->{headerAdded} = 1;
    } else {
        my $fields = $table->{fields};

        if ($fields) {
            $row = {map {
                my $i = $_;
                my $field = $fields->[$i];
                $field = $i unless defined $field;
                ($field => $row->[$i]);
            } 0..$#$row};
        }

        push @{$table->{rows}}, $row;
    }
}

sub _makeTable {
    my ($table, $param) = @_;

    my $fields = $table->{fields};
    my $rows = $table->{rows};

    if (my $callback = $param->{callback}) {
        $rows = _applyCallback($callback, $rows, $param);
    }

    if ($fields) {
        return {fields => $fields, rows => $rows};
    } else {
        return $rows;
    }
}

sub _normalizeTable {
    my ($table, $param) = @_;

    my $fields;
    my $rows = [];

    if (ref $table eq 'HASH') {
        $fields = $table->{fields} || [];
        $rows = $table->{rows} || [];
    } elsif (ref $table eq 'ARRAY') {
        $rows = $table;
    }

    if (my $callback = $param->{callback}) {
        $rows = _applyCallback($callback, $rows, $param);
    }

    for my $row (@$rows) {
        if (ref $row eq 'HASH') {
            $row = [map {$row->{$_}} @$fields] if $fields;
        }
    }

    return {
        fields => $fields,
        rows   => $rows,
    };
}

sub parseTable {
    my $param = _parseParam(\@_, ['text']);

    my $table = {
        header => TWiki::isTrue($param->{header}, 1),
        fields => _parseFields($param->{fields}),
        rows   => [],
    };

    my $text = $param->{text};
    my $target = $param->{target} || 1;
    my $current = 0;
    my $detected = 0;

    while ($text =~ /^(.*)$/gm) {
        my $line = $1;

        if ($line =~ /^\s*\|\s*((.|\\\n)*?)\s*\|\s*$/) {
            my $values = [map {
                s/^\*(.*)\*$/$1/;
                unescapeTable($_);
            } split(/\s*\|\s*/, $1)];

            if (!$detected) {
                $current++;
                $detected = 1;
            }

            if ($current == $target) {
                _addToTable($table, $values);
            }
        } else {
            if ($detected) {
                $detected = 0;
                last if $current >= $target;
            }
        }
    }

    return _makeTable($table, $param);
}

sub formatTable {
    my $param = _parseParam(\@_, ['table?']);
    my $table = $param->{table} || $param;
    my $header = TWiki::isTrue($param->{header}, 1);

    $table = _normalizeTable($table, $param);
    my $fields = $table->{fields};
    my $rows = $table->{rows};

    my @result;

    if ($fields && $header) {
        push @result, '| *'.join('* | *', @$fields).'* |';
    }

    for my $row (@$rows) {
        if (!ref $row) {
            push @result, $row;
        } else {
            push @result, '| '.join(' | ', map {escapeTable($_)} @$row).' |';
        }
    }

    return join("\n", @result);
}

sub parseCSV {
    my $param = _parseParam(\@_, ['text']);

    my $table = {
        header => TWiki::isTrue($param->{header}, 1),
        fields => _parseFields($param->{fields}),
        rows   => [],
    };

    my $csv = Text::CSV->new({binary => 1});
    my $in = IO::String->new($param->{text});

    while (my $row = $csv->getline($in)) {
        _addToTable($table, $row);
    }

    return _makeTable($table, $param);
}

sub formatCSV {
    my $param = _parseParam(\@_, ['table?']);
    my $table = $param->{table} || $param;
    my $header = TWiki::isTrue($param->{header}, 1);

    $table = _normalizeTable($table, $param);
    my $fields = $table->{fields};
    my $rows = $table->{rows};

    my $csv = Text::CSV->new({eol => "\n"});
    my $out = IO::String->new(\my $text);

    if ($fields && $header) {
        $csv->print($out, $fields);
    }

    for my $row (@{$table->{rows}}) {
        $csv->print($out, $row);
    }

    return $text;
}

sub _makeRequestParams {
    my ($param) = @_;
    my $headers = $param->{headers} || [];
    my $options = {};

    if (ref($headers) eq 'HASH') {
        $headers = [%$headers];
    }

    my $param2option = {
        method      => 'method',
        timeout     => 'timeout',
        maxRedirect => 'max_redirect',
    };

    for my $paramKey (keys %$param2option) {
        my $optionKey = $param2option->{$paramKey};
        $options->{$optionKey} = $param->{$paramKey} if defined $param->{$paramKey};
    }

    if (defined $param->{insecureSSL}) {
        $options->{ssl_opts} ||= {};
        $options->{ssl_opts}{verify_hostname} = !TWiki::isTrue($param->{insecureSSL});
    }

    # Cookies should persist throughout the EJS session
    if (!$CONTEXT->{cookies}) {
        eval "use HTTP::Cookies";
        $CONTEXT->{cookies} = $@ ? {} : HTTP::Cookies->new();
    }

    $options->{cookie_jar} = $CONTEXT->{cookies};

    return ($headers, $options);
}

sub _makeResponse {
    my ($res) = @_;
    my $headers = {};

    if ($res->isa('TWiki::Net::HTTPResponse')) {
        $headers = $res->{headers};
    } elsif ($res->isa('HTTP::Response')) {
        # Use RFC-suggested cases (See HTTP::Headers doc)
        for my $name ($res->header_field_names) {
            my @values = $res->header($name);
            $headers->{$name} = @values != 1 ? \@values : $values[0];
        }
    }

    my $code = $res->code;

    return {
        content    => $res->content,
        code       => $code,
        message    => $res->message,
        headers    => $headers,
        isSuccess  => $code < 300 ? 1 : 0,
        isRedirect => $code >= 300 && $code < 400 ? 1 : 0,
        isError    => $code >= 400 ? 1 : 0,
    };
}

sub getExternalResource {
    my $param = _parseParam(\@_, ['url']);
    my ($headers, $options) = _makeRequestParams($param);
    my $res = TWiki::Func::getExternalResource($param->{url}, $headers, $options);
    return _makeResponse($res);
}

sub postExternalResource {
    my $param = _parseParam(\@_, ['url', 'content']);
    my ($headers, $options) = _makeRequestParams($param);
    my $res = TWiki::Func::postExternalResource($param->{url}, $param->{content}, $headers, $options);
    return _makeResponse($res);
}

sub captureOutput {
    my $callback = shift;
    local $CAPTURE = [];
    $callback->(@_);
    return join('', @$CAPTURE);
}

sub getJavaScriptEngine {
    my $class = ref $CONTEXT->{adapter};
    $class =~ s/^EJS::Template::JSAdapter:://;
    return $class;
}

sub getEJSConfig {
    return $CONTEXT->{config};
}

sub executeEJS {
    my ($text, $variables) = @_;
    _processInlineEJS(\$text, $variables);
    return 1;
}

my $json = JSON->new->allow_nonref->allow_blessed;

sub JSON_stringify {
    # Do not use $json->encode(), which would unexpectedly "call"
    # subroutine/function type values within the object
    my $ref = ref $_[0];

    my @result;

    if (!$ref) {
        if (!defined $_[0]) {
            return 'null';
        } elsif (looks_like_number $_[0]) {
            return $_[0];
        } else {
            @result = ('"', escapeQuote($_[0]), '"');
        }
    } elsif ($ref eq 'HASH') {
        my $first = 1;
        push @result, '{';

        for my $key (sort keys %{$_[0]}) {
            if ($first) {
                $first = 0;
            } else {
                push @result, ', ';
            }

            push @result, '"', escapeQuote($key), '"';
            push @result, ': ';
            push @result, JSON_stringify($_[0]{$key});
        }

        push @result, '}';
    } elsif ($ref eq 'ARRAY') {
        my $first = 1;
        push @result, '[';

        for my $value (@{$_[0]}) {
            if ($first) {
                $first = 0;
            } else {
                push @result, ', ';
            }

            push @result, JSON_stringify($value);
        }

        push @result, ']';
    } elsif ($ref eq 'CODE') {
        push @result, '"[code]"';
    } else {
        push @result, 'null';
    }

    return join('', @result);
}

sub JSON_parse {
    my ($text) = @_;
    $text = $text->value if ref($text) =~ /^JE::/;
    my $obj = $json->decode($text);
    return $obj;
}

sub _consoleOutput {
    my ($method, @objects) = @_;

    EJS::Template->context->print(
        "<script>console.$method(",
        join(', ', map {JSON_stringify($_)} @objects),
        ");</script>"
    );

    return 1;
}

sub console_log {
    _consoleOutput('log', @_);
}

sub console_debug {
    _consoleOutput('debug', @_);
}

sub console_info {
    _consoleOutput('info', @_);
}

sub console_warn {
    _consoleOutput('warn', @_);
}

sub console_error {
    _consoleOutput('error', @_);
}

1;
