# Module of TWiki Enterprise Collaboration Platform, http://TWiki.org/
#
# Copyright (C) 2013-2014 Wave Systems Corp.
# Copyright (C) 2013-2015 Peter Thoeny, peter[at]thoeny.org
# Copyright (C) 2013-2015 TWiki Contributors. All Rights Reserved.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 3
# of the License, or (at your option) any later version. For
# more details read LICENSE in the root of this distribution.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
# As per the GPL, removal of this notice is prohibited.

=begin twiki

This package adds an autocomplete input field to TWiki forms. 

=cut

package TWiki::Plugins::AutocompletePlugin;

use strict;

require TWiki::Func;    # The plugins API

# ========================================================
our $VERSION = '$Rev: 29252 (2015-05-29) $';
our $RELEASE = '2015-05-29';
our $SHORTDESCRIPTION = "Autocomplete input field, for use in TWiki forms and TWiki applications";

our $NO_PREFS_IN_TOPIC = 1;
our @allowedAttributes = (
    'form', 'id', 'max', 'maxlength', 'min', 'onblur', 'onfocus', 'onchange', 'onselect',
    'onmouseover', 'onmouseout', 'pattern', 'placeholder', 'spellcheck', 'tabindex',
    'title', 'translate'
  );
our $minOptionsShown = 1;
our $maxOptionsShown = 6;
our $headerDone;
our $header = <<'HERE';
<style>
.autocompleteSelect {
  width: 99%;
  border: 1px solid #666;
  box-shadow: 1px 1px 4px #666;
}
.autocompleteDropdownImage {
  vertical-align: middle;
  margin-left: -13px;
}
</style>
<script type="text/javascript">
// define .is(':focus') in case of jQuery 1.5 and older
(function ( $ ) {
  var filters = $.expr[":"];
  if( !filters.focus ) { 
    filters.focus = function( elem ) {
      return elem === document.activeElement && ( elem.type || elem.href );
    };
  }
})( jQuery );
var autocompleteOptionsHash = {};
function saveAllAutocompleteOptions( sel ) {
  autocompleteOptionsHash[sel.attr('name')] = sel.html();
}
function restoreAllAutocompleteOptions( sel ) {
  if( sel.attr('name') in autocompleteOptionsHash ) {
    sel.html( autocompleteOptionsHash[sel.attr('name')] );
    return true;
  }
  return false;
}
$(document).ready(function() {
  $('.autocompleteTextField').focus(function( e ) {
    var sel = $(this).parent().parent().find('.autocompleteSelect');
    saveAllAutocompleteOptions( sel );
  })
  .blur(function( e ) {
    var sel = $(this).parent().parent().find('.autocompleteSelect');
    // Due to blur/focus sequence, we need to use timer to test if new focus is in select
    var txtObj = $(this);
    setTimeout(function(){
     if( sel.is(':focus') ) {
        // do nothing
      } else {
        restoreAllAutocompleteOptions( sel );
        sel.parent().css({display: 'none'});
        var txt = txtObj.val();
        if( txt.length > 0 && sel.hasClass( 'autocompleteSelectRestricted' ) ) {
          txt = txt.replace(/[-[\]{}()*+?.,\\^$|#\s]/g, "\\$&");
          var reFull  = new RegExp( '^'+txt+'$', 'i' );
          var reStart = new RegExp( '^'+txt, 'i' );
          var found = '';
          sel.find('option').each(function() {
            var optVal = this.value;
            var optTxt = $(this).text();
            if( optTxt.match( reFull ) ) {
              found = optVal;
              return;
            } else if( !found && ( optTxt.match( reStart ) || optVal.match( reStart ) ) ) {
              found = optVal;
              // & keep looking
            }
          });
          txtObj.val(found);
        }
      }
    }, 100);
  });
  $(document).delegate('.autocompleteTextField', 'keyup', function( e ) {
    var sel = $(this).parent().parent().find('.autocompleteSelect');
    var txt = $(this).val().toLowerCase();
    if( txt.length>0 ) {
      if( restoreAllAutocompleteOptions( sel ) ) { 
        sel.find('option').each(function() {
          if( $(this).text().toLowerCase().indexOf(txt) < 0 ) {
            $(this).remove();
          }
        });
        var nOpt = sel.find('option').length;
        if( nOpt < 1 ) {
          sel.parent().css({display: 'none'});
        } else {
          if( nOpt < 2 ) { nOpt = 2; }
          if( nOpt > 6 ) { nOpt = 6; }
          sel.attr('size', nOpt);
          sel.parent().css({display: 'inline'});
        }
      }
    } else {
      sel.parent().css({display: 'none'});
    }
  });
  $('.autocompleteDropdownImage').click(function( event ) {
    event.preventDefault();
    if( $(this).parent().find('.autocompleteTextField').is(':focus') ) {
      // do nothing
    } else { 
      var sel = $(this).parent().parent().find('.autocompleteSelect');
      restoreAllAutocompleteOptions( sel );
      sel.val($('.autocompleteTextField').val());
      var nOpt = sel.find('option').length;
      if( nOpt < 2 ) { nOpt = 2; }
      if( nOpt > 6 ) { nOpt = 6; }
      sel.attr('size', nOpt);
      sel.parent().css({display: 'inline'});
      sel.focus();
    }
  });
  $('.autocompleteSelect').click(function() {
    var txtObj = $(this).parent().parent().find('.autocompleteTextField');
    var selectedTxt = $(this).find('option:selected').val();
    $(this).parent().css({display: 'none'});
    restoreAllAutocompleteOptions( $(this) );
    txtObj.val(selectedTxt).focus();
  })
  .blur(function() {
    var txtObj = $(this).parent().parent().find('.autocompleteTextField');
    $(this).parent().css({display: 'none'});
    restoreAllAutocompleteOptions( $(this) );
    txtObj.focus();
  });
  $(document).delegate('.autocompleteSelect', 'keyup', function( e ) {
    var txtObj = $(this).parent().parent().find('.autocompleteTextField');
    if( e.which==13 || e.which==27 ) {
      if( e.which==13 ) {
        txtObj.val($(this).find('option:selected').val());
      }
      $(this).parent().css({display: 'none'});
      restoreAllAutocompleteOptions( $(this) );
      txtObj.focus();
    } 
  });
});
</script>
HERE

# ========================================================
sub initPlugin {
  my( $topic, $web, $user, $installWeb ) = @_;

  TWiki::Func::registerTagHandler('AUTOCOMPLETE', \&handleAutocomplete );
  $headerDone = 0;

  return 1;
}

# ========================================================
sub handleAutocomplete  {
  my ( $session, $params ) = @_;

  my $name    = $params->{name} || 'missing_name';
  my $value   = $params->{value} || '';
  my $options = $params->{options} || '';
  my $size    = $params->{size} || 40;
  my $type    = $params->{type} || '';

  return renderForEdit( $name, $value, $options, $size, $type, $params );
}

# ========================================================
sub renderForEdit {
    my( $name, $value, $options, $size, $type, $params ) = @_;

    unless( $headerDone ) {
        TWiki::Func::addToHEAD( 'AUTOCOMPLETEPLUGIN', $header );
        $headerDone = 1;
    }

    my %optionParams = ( class => 'twikiEditFormOption' );
    my @optionList = split( /, */, $options );
    my $choices = '';
    foreach my $option ( @optionList ) {
        $option =~ s/<nop/&lt\;nop/go;
        if( $type =~ /values?/ && $option =~ /^(.*?): ?(.+)$/ ) {
            $optionParams{value} = $1;
            $option = $2;
        } else {
            delete $optionParams{value};
        }
        $choices .= CGI::option( \%optionParams, $option );
    }
    my $optionsShown = scalar( @optionList );
    if( $optionsShown > $maxOptionsShown ) {
        $optionsShown = $maxOptionsShown;
    } elsif( $optionsShown < $minOptionsShown ) {
        $optionsShown = $minOptionsShown;
    }

    $size ||= 40;
    my $width = sprintf( "%1.2fem", $size * 0.57 ); # approximate number of chars
    my $style = "width: $width";
    $style .= '; ' . $params->{style} if( $params && $params->{style} );
    $style .= ';' if( $style !~ /;$/ );
    my $classes = 'twikiInputField twikiEditFormTextField autocompleteTextField';
    $classes .= ' twikiMandatory' if( TWiki::Func::isTrue( $params && $params->{mandatory} ) );
    $classes .= ' ' . $params->{class} if( $params && $params->{class} );
    my %textParams = (
        -class => $classes,
        -name  => $name,
        -size  => $size,
        -value => $value,
        -style => $style,
        -autocomplete => 'off',
      );
    foreach my $attr ( @allowedAttributes ) {
        if( $params && defined $params->{$attr} ) {
            $textParams{$attr} = $params->{$attr};
        }
    }
    $style = '';
    if( $params && $params->{containerstyle} ) {
        $style = $params->{containerstyle};
        $style .= ';' if( $style !~ /;$/ );
        $style = " style='$style'";
    }
    my $text = "<div class='autocompleteContainer'$style>";
    $text .= "<div style='white-space: nowrap;'>";
    $text .= CGI::textfield( \%textParams );
    $text .= CGI::image_button(
        -class   => 'autocompleteDropdownImage',
        -name    => $name . '_Image',
        -src     => TWiki::Func::getPubUrlPath() . '/' .
                    TWiki::Func::getTwikiWebname() . '/AutocompletePlugin/pick.gif',
        -alt     => 'Select',
        -title   => 'Select',
      );
    $text .= "</div>";
    $width = sprintf( "%1.2fem", $size * 0.57 + 0.7 );
    $style = "position: absolute; width: $width; z-index: 10; display: none";
    $text .= "<div style='$style'>";
    $classes = 'twikiSelect twikiEditFormSelect autocompleteSelect';
    if( $type =~ /restrict/ ) {
        $classes .= ' autocompleteSelectRestricted';
    }
    my $selectParams = {
        -class   => $classes,
        -name    => $name . '_pick',
        -size    => $optionsShown,
    };
    if( $params && $params->{selectstyle} ) {
        $style = $params->{selectstyle};
        $style .= ';' if( $style !~ /;$/ );
        $selectParams->{style} = $style;
    }
    $text .= CGI::Select( $selectParams, $choices );
    $text .= '</div></div>';

    return $text;
}

# ========================================================
1;
