# Plugin for TWiki Enterprise Collaboration Platform, http://TWiki.org/
#
# Copyright (C) 2020-2021 Peter Thoeny, peter[at]thoeny.org 
# Copyright (C) 2020-2021 TWiki Contributors. All Rights Reserved.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version. For
# more details read LICENSE in the root of this distribution.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details, published at
# http://www.gnu.org/copyleft/gpl.html
#
# As per the GPL, removal of this notice is prohibited.

package TWiki::Plugins::AnchorLinkPlugin;

# =========================
our $VERSION = '$Rev: 31001 (2021-03-25) $';
our $RELEASE = '2021-03-25';
our $SHORTDESCRIPTION = 'Show anchor links on hover over headings, to copy anchor link address';
our $NO_PREFS_IN_TOPIC = 1;

# =========================
our $debug = $TWiki::cfg{Plugins}{AnchorLinkPlugin}{Debug} || 0;
our $web;
our $topic;
our $headerDone;
our @headingList;

# =========================
sub initPlugin {
    ( $topic, $web ) = @_;

    # Plugin correctly initialized
    TWiki::Func::writeDebug( "- AnchorLinkPlugin::initPlugin( $web.$topic ) is OK" ) if( $debug );

    TWiki::Func::registerTagHandler( 'ANCHORLINK', \&_ANCHORLINK );

    @headingList = grep { $_ =~ /^h[1-6]$/i }
      split( /,\s*/, TWiki::Func::getPreferencesValue('ANCHORLINKHEADINGS') ||
        $TWiki::cfg{Plugins}{AnchorLinkPlugin}{Headings} || 'h2, h3, h4' );

    if(scalar ( @headingList )) {
        _addHeader();
    }
    return 1;
}

# =========================
sub _addHeader {
    return if $headerDone;
    $headerDone = 1;

    my $headings = join( ', ', @headingList );
    _writeDebug( "_addHeader($headings)" );
    my $html .= <<'EOF';

<style type="text/css" media="all">
.twikiALP {
    position: absolute;
    left: 15px;
    width: 32px;
    opacity: 0;
}
.twikiALP a:link {
    border: 0 none;
}
.twikiALPParent:hover .twikiALP {
    opacity: 1;
}
</style>
<script type="text/javascript">
// <![CDATA[
$('document').ready(function() {
    $('%HEADINGS%').each(function(i, elem) {
        var $elem = $(elem);
        var anchor = $elem.find('a').first().attr('name');
        if(anchor) {
            var html = '<div class="twikiALP" data-anchor="' + anchor + '"></div>';
            $elem.prepend(html);
        }
    });
    $('.twikiALP').each(function(i, elem) {
        var $elem = $(elem);
        var anchor = $elem.data('anchor');
        var attrs = 'href="#' + anchor + '"';
        if($elem.data('add')) {
            attrs += ' name="' + anchor + '"';
        }
        if(anchor) {
            var html = '<a ' + attrs + '>'
              + '<img src="%PUBURLPATH%/%SYSTEMWEB%/AnchorLinkPlugin/anchor-bg.gif"'
              + ' width="16" height="16" border="0" /></a>';
            $elem.html(html);
            var $parent = $elem.parent();
            var parentTag = $parent.prop('tagName');
            var $prev;
            var prevTag = '';
            var found = false;
            while(true) {
                if(parentTag.match(/^(TR|TABLE|LI|H[1-6])$/)) {
                    found = true;
                    break;
                } else if((parentTag == 'DIV' && $parent.hasClass('patternTopic')) || parentTag == 'BODY') {
                    // at top level
                    if(prevTag) {
                        // use previous parent
                        $parent = $prev;
                        parentTag = prevTag;
                        found = true;
                    } else {
                        // ignore top level paragraphs
                        // TWiki has weird <p></p>Paragraph 1<p></p>Paragraph 2<p></p>
                        // FIXME: Enclose paragraph in div, and set parent to that
                        //var textBefore = $elem[0].previousSibling.nodeValue;
                        //var textAfter = $elem[0].nextSibling.nodeValue;
                    }
                    break;
                }
                $prev = $parent;
                $parent = $parent.parent();
                prevTag = parentTag;
                parentTag = $parent.prop('tagName');
            }
            if(found) {
                var offset = $elem.offset();
                offset.top = $parent.offset().top;
                var height = 25; // average line height
                if(parentTag.match(/^H[1-6]/)) {
                    height = $parent.height() + 10;
                }
                if(height > 16) {
                    var padding = (height - 16) / 2;
                    offset.top = offset.top + padding;
                }
                offset.top = Math.round(offset.top * 10) / 10;
                $elem.offset(offset);
                $parent.addClass('twikiALPParent');
            }
        }
    });
    $('.twikiALP a').click(function() {
        $('html, body').unbind().animate({
            scrollTop: $(this).offset().top - 20
        }, 'slow');
    });
});
// ]]>
</script>
EOF
    $html =~ s/%HEADINGS%/$headings/;
    TWiki::Func::addToHEAD( 'ANCHORLINKPLUGIN', $html );
}

# =========================
sub _ANCHORLINK {
    my ( $session, $params, $theTopic, $theWeb, $meta ) = @_;
    my $anchor = $params->{_DEFAULT} || '';
    $anchor =~ s/[^a-zA-Z0-9_\-]//g;
    _writeDebug( "_ANCHORLINK( $anchor )" );
    return '' unless( $anchor );
    my $html = '<div class="twikiALP"' . ' data-anchor="' . $anchor . '" data-add="1"></div>';
    return $html;
}

# =========================
sub _writeDebug
{
    my ( $msg ) = @_;
    return unless( $debug );
    TWiki::Func::writeDebug( "- AnchorLinkPlugin::$msg" );
}

# =========================
1;
