# TWiki Enterprise Collaboration Platform, http://TWiki.org/
#
# Copyright (C) 2005 Peter Thoeny, peter@thoeny.com
# and TWiki Contributors. All Rights Reserved. TWiki Contributors
# are listed in the AUTHORS file in the root of this distribution.
# NOTE: Please extend that file, not this notice.
#
# Additional copyrights apply to some or all of the code in this
# file as follows:
# Copyright (C) 2005 Greg Abbas, twiki@abbas.org
# Copyright (C) 2005 Harald Joerg, haj@oook.m.uunet.de
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version. For
# more details read LICENSE in the root of this distribution.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
# As per the GPL, removal of this notice is prohibited.

=pod

---+ package TWiki::Client::RegistrationOnDemandLogin

This is a login manager that you can specify in the security setup 
section of [[%SCRIPTURL%/configure%SCRIPTSUFFIX%][configure]].

This login manager is intended for use in cases where every user is
authenticated by the web server before using TWiki for the first time.
In this cases, TWiki will encounter users with =$ENV{REMOTE_USER}= set,
who, in contrast to the assumptions of ApacheLogin.pm, never have
registered before.

The login manager is sort of a parallel track to TemplateLogin:
TemplateLogin assumes that you are a registered user and allows you
to login on demand, whereas RegistrationOnDemandLogin assumes that you
did login (note that you can't 'logout' in a HTTP sense) and allows
you to register on demand if you start an action for which TWiki
requires an 'authenticated' session.

TWikiUserAuthentication is of not much help here....

TWiki::Client::RegistrationOnDemandLogin is a subclass of TWiki::Client; see that 
class for documentation of the methods of this class.

=cut

package TWiki::Client::RegistrationOnDemandLogin;

use strict;
use Assert;
#use Data::Dumper;  # for DEBUGGING
use TWiki::UI::Register;

BEGIN {
    use vars qw(
	$field_hidden
	$field_edit
	$blur_last
	$blur_full
    );

    $field_hidden = qq[<input type="hidden" name="%s" value="%s" /><B>%s</B>];
    $field_edit   = qq[<input type="text" name="%s" value="%s" size="40" onblur="%s" />];
    $blur_last  = qq[this.form.Twk1WikiName.value=this.form.Twk1FirstName.value+this.value ];
    $blur_full  = qq[var sIn = this.value; var sOut = ''; var chgUpper = true; for ( var i = 0; i < sIn.length; i++ ) { var ch = sIn.charAt( i ); var ch2 = ''; if((ch=='&#224;')||(ch=='&#225;')||(ch=='&#226;')||(ch=='&#227;')||(ch=='&#229;') ) { ch = 'a'; } if( (ch=='&#232;')||(ch=='&#233;')||(ch=='&#234;') ) { ch = 'e'; } if( ch=='&#231;') { ch = 'c'; } if( ch=='&#241;') { ch = 'n'; } if( (ch=='&#242;')||(ch=='&#243;')||(ch=='&#244;') ) { ch = 'o'; } if( ch=='&#228;') { ch = 'a'; ch2 = 'e'; } if( ch=='&#246;') { ch = 'o'; ch2 = 'e'; } if( (ch=='&#249;')||(ch=='&#250;')||(ch=='&#251;') ) { ch = 'o'; } if( ch=='&#252;') { ch = 'u'; ch2 = 'e'; } if( ch=='&#196;') { ch = 'A'; ch2 = 'e'; } if( ch=='&#214;') { ch = 'O'; ch2 = 'e'; } if( ch=='&#220;') { ch = 'U'; ch2 = 'e'; } if( ch=='&#223;') { ch = 's'; ch2 = 's'; } if( ((ch>='a')&&(ch<='z')) || ((ch>='A')&&(ch<='Z')) ) { if( chgUpper ) { ch = ch.toUpperCase(); chgUpper = false; } sOut+=ch; if( ch2!='') { sOut+=ch2; } } else { if( ch==' ' ) { chgUpper = true; } } } this.form.Twk1WikiName.value=sOut;];

} # BEGIN


@TWiki::Client::RegistrationOnDemandLogin::ISA = qw( TWiki::Client );

sub new {
    my( $class, $twiki ) = @_;
    my $this = bless( $class->SUPER::new($twiki), $class );
    $twiki->enterContext( 'can_login', 1 );
    return $this;
}

sub forceAuthentication {
    my $this = shift;
    my $twiki = $this->{twiki};

#    print STDERR "forceAuthentication\n";

    unless( $twiki->inContext( 'authenticated' )) {
        my $query = $twiki->{cgiQuery};
        #This method of encoding the parameters could fall foul of the
        # limit on parameters to a GET, since a redirect is used later.
        # Fortunately this is very unlikely, because of the way TWiki works.
        my $origurl = $query->url() . $query->path_info() .
          '?' . $query->query_string();
        $query->param( origurl => $origurl );
        $this->_register_on_demand( $query, $twiki );
        return 1;
    }
    return undef;
}

# SMELL: I haven't found any callers for that method.
sub DISABLED_loginUrl {
    my $this = shift;
    my $twiki = $this->{twiki};
    my $topic = $twiki->{topicName};
    my $web = $twiki->{webName};
    return $twiki->getScriptUrl( $web, $topic, 'logon', @_ );
}

sub getUser {
    my $this = shift;
    my $twiki = $this->{twiki};
    # We need the *real* thing, as close as possible.  Using the remoteUser
    # field from the $twiki object just makes sure that there's a default.
    # SMELL: Am I the only one to use this field?
    my $remote_user  =  $twiki->{remoteUser};

#    print STDERR "RODL:getUser($remote_user)\n";

    # This is actually cheating:  If a user is registered then he should be
    # found by findUser.  Otherwise *pretend* we are logged in as guest
    # so that lateron the session gets marked as "unauthenticated"
    # by our caller.
    my $user         =  $twiki->{users}->findUser($remote_user,undef,1)
                     || $twiki->{users}->findUser($TWiki::cfg{DefaultUserLogin});
    return $user->login();
}


# FIXME: This is just a copy from TemplateLogin - I've no idea what happens here.
sub checkSession {
    my $this = shift;
    my $cgisession = $this->{cgisession};
    my $query = $this->{twiki}->{cgiQuery};
    my $authUserSessionVar = $TWiki::Client::authUserSessionVar;

    $cgisession->clear() if(
        defined($cgisession) && defined($cgisession->param) &&
        defined($query) && defined( $query->remote_user() ) &&
        defined($authUserSessionVar) &&
        defined( $cgisession->param( $authUserSessionVar ) ) &&
        "" ne $query->remote_user() &&
        "" ne $cgisession->param( $authUserSessionVar ) &&
        $query->remote_user() ne $cgisession->param( $authUserSessionVar ) );
}



sub _build_field {
    my($field, $value, $edit, $on_blur) = @_;
    
    $on_blur = '' if( !defined($on_blur) ); # avoid the uninitialized warnings
    $edit ||= '';
    return( ($edit ne '') ?
	    sprintf($field_edit,   $field, $value, $on_blur) :
	    sprintf($field_hidden, $field, $value, $value) );
} #_build_field


# This routine is called if forceAuthentication detects the need of a
# registration on demand.  It collects the form data for minimal
# registration from a template and prints the query.
sub _register_on_demand {
    my( $this, $query, $session ) = @_;

#    print STDERR "_register_on_demand\n";

    my $twiki     =  $this->{twiki};
    my $templates = $twiki->{templates};
    my $template  =  $templates->readTemplate('register', $twiki->getSkin() );
    my $origurl   =  $query->param( 'origurl' ) || ''; # avoid undef warnings
    my $banner    =  $templates->expandTemplate( 'REGISTER_BANNER' );
    my $topic     =  $twiki->{topicName};
    my $web       =  $twiki->{webName};

# BixToken specific changes
    my($first, $mid, $last, $wname, $email, $remote, $exists, $backup);
    $remote = $ENV{REMOTE_USER};
    $wname  = $ENV{BixToken_name};
    $email  = $ENV{BixToken_mail};

    ($first, $mid, $last) = split(/\./, $email);
    $backup = join('', ucfirst($first), ucfirst($mid), ucfirst($last));

    if( defined($wname) && $wname =~ m/^(.*)\s+(.*)$/ ){
	$first = $1;
	$mid   = '';
	$last  = $2;
	$wname = join('', ucfirst($first), ucfirst($last));
	if( $session->{store}->topicExists( $TWiki::cfg{UsersWebName}, $wname) ){
	    $wname = $backup;
	}
    } else {
	$wname = $backup;
    }

    print STDERR "RegistrationOnDemand [$remote][$wname][$email][$first][$mid][$last]\n";

    if( defined($wname) && $wname ne '' ){
	$wname = _build_field('Twk1WikiName',  $wname);
	$first = _build_field('Twk1FirstName', $first);
	$last  = _build_field('Twk1LastName',   $last);
    } else {
	$first = _build_field('Twk1FirstName', '',     1, '');
	$last  = _build_field('Twk1LastName',  '',     1, $blur_last);
	$wname = _build_field('Twk1WikiName',  $wname, 1, $blur_full);
    }

    if( $email ne '' ){
	$email = _build_field('Twk1Email', $email);
    } else {
	$email = _build_field('Twk1Email', '', 1, '');
    }

    if( $remote ne '' ){
	$remote = _build_field('Twk1LoginName', $remote);
    } else {
	$remote = _build_field('Twk1LoginName', '', 1, '');
    }

    print STDERR "RegistrationOnDemand, after[$first][$last][$wname][$email]\n";

    $template =~ s/%FIRSTNAME%/$first/gs;
    $template =~ s/%LASTNAME%/$last/gs;
    $template =~ s/%FULLNAME%/$wname/gs;
    $template =~ s/%EMAILADDR%/$email/gs;
    $template =~ s/%REMOTEUSER%/$remote/gs; 
    $template =~ s/%ORIGURL%/$origurl/gs;

# END  BixToken

    $template =~ s/%BANNER%/$banner/gs;

    $template = $twiki->handleCommonTags( $template, $web, $topic );
    $template = $twiki->{renderer}->getRenderedVersion( $template, '' );
    $template =~ s/<nop>//gs;

    $twiki->writePageHeader( $query );

    print STDOUT $template;
}

=pod

---++ ObjectMethod login( $query, $twiki )

Derived from TemplateLogin.pm, the form created by _register_on_demand
has bin/login as the "action" attribute.  The little stub in bin/login
just Does The Right Thing to end up in the login manager where the
form originated.

This routine does a registration on the fly:

   1 Check for a wikiname.  If we haven't got one just redisplay the form.
   2 Validate the parameters (as in usual registration)
   3 Create the user in the twiki session's user hash
   4 Create the user's home topic
   5 Add the user to TWiki's UserList
   6 Mark the session as "authenticated"
   7 Finally, redirect to the action the user was about to execute

=cut


sub login {
    my( $this, $query, $session ) = @_;

    my $twiki = $this->{twiki};
    my $data  = TWiki::UI::Register::_getDataFromQuery( $query );

#    print STDERR "_login Data: ", Dumper($data), "\n";
    my $wikiname  = $data->{WikiName};
    if( defined($wikiname) ){
        my $topic    = $twiki->{topicName};
        TWiki::UI::Register::_validateRegistration ( $twiki, $data, $query, $topic );

        # validation throws an error, so we seem to have succeeded so far

        my $login     = $twiki->{remoteUser} || $data->{LoginName};
	my $web       = $twiki->{webName};

#	print STDERR "login: [$login] [$web]", Dumper($data), "\n";

	# create the user's home page
	$data->{webName}  =  $TWiki::cfg{UsersWebName};

	my(%row, %settings);
	%row = (
		webName   => $TWiki::cfg{UsersWebName},
		WikiName  => $data->{WikiName},
		Email     => $data->{Email},
		FirstName => $data->{FirstName},
		LastName  => $data->{LastName},
		LoginName => $data->{LoginName},
		Password  => 'NoPassword',
		);
	%settings = (
		     doOverwriteTopics => 0,
		     doEmailUserDetails  => 1,
		     fieldNames => [ qw(WikiName Email FirstName LastName LoginName) ],
		     );

#	print STDERR "before _registerSingleBulkUser: ", Dumper(\%row), Dumper(\%settings), "\n";

	my($usertopic, $log);
	($usertopic, $log) = TWiki::UI::Register::_registerSingleBulkUser($session,
									   \%row,
									   \%settings );
        $this->userLoggedIn( $login );

	my $origurl  = $query->param( 'origurl' );
#	print STDERR "ORIG[$origurl]\n";
        if( !$origurl || $origurl eq $query->url() ) {
            $origurl = $twiki->getScriptUrl( $web, $topic, 'view' );
        }
        $this->redirectCgiQuery( $query, $origurl );
        return;
    } else {
        _register_on_demand(@_);
    }
}

1;

